/* ---------------------------------------------------------- 
%   (C)1993, 1994, 1995 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
%   (C)1996, 1997, 1998, 1999 Japan Information Processing Development Center
%       (Read COPYRIGHT-JIPDEC for detailed information.)
----------------------------------------------------------- */

#include <assert.h>
#include <klic/basic.h>  /* fatal, fatalf */
#include <klic/struct.h>

/* klic_interrupt, malloc_check,
 * enqueue_goal, enqueue_after_waiting, enqueue_throw_goal
 */
#include <klic/primitives.h>

#include <stdio.h>

#include "trace.h"  /* enqueue_trace_rec, trace_enqueued_goals, trace_flag */

#ifdef SHM
#include <setjmp.h>
#include "shm.h"

extern jmp_buf klic_topmost;

static module mastersucceed(struct global_variables* glbl, struct goalrec* qp,
  q* allocp, const struct predicate* toppred );

static module childsucceed(struct global_variables* glbl, struct goalrec* qp,
  q* allocp, const struct predicate* toppred );

static const struct predicate mastersucceed_pred = { mastersucceed, 0, 0 };
static const struct predicate childsucceed_pred = { childsucceed, 0, 0 };
#endif  /* SHM */

#define new_prioqrec(pqr) \
do{ \
  struct prioqrec* temp; \
  while( (temp = prioqrec_free) == NULL ){ \
    prioqrec_free = more_prioqrec(); \
  } \
  prioqrec_free = temp->next; \
  (pqr) = temp; \
}while(0)

#define free_prioqrec(pqr) \
do{ \
  pqr->next = prioqrec_free; \
  prioqrec_free = pqr; \
}while(0)

extern char* calloc();
extern module topsucceed();

#ifdef DIST
extern struct goalrec* send_throw_goal();
#endif
#ifdef SHM
extern struct goalrec* throw_goal_routine();
#endif

/* temp */
extern struct ex_goalrec** shm_qp;


extern module wait_prio_routine(struct global_variables* glbl,
  struct goalrec* qp, q* allocp, const struct predicate* toppred );

extern module wait_penum_routine(struct global_variables* glbl,
  struct goalrec* qp, q* allocp, const struct predicate* toppred );

static module queue_empty(struct global_variables* glbl, struct goalrec* qp,
  q* allocp, struct goalrec* fg, const struct predicate* toppred);

const struct predicate queue_empty_pred = { queue_empty, 0, 0 };
const struct predicate topsucceed_pred = { topsucceed, 0, 0 };

struct goalrec goal_queue_tail = { 0, &queue_empty_pred, { 0 } };


static unsigned long current_prio0;	/* current priority */
extern unsigned long current_prio(void){ return current_prio0; }
extern void set_current_prio(unsigned long prio){ current_prio0 = prio; }

static struct prioqrec prioq0;	/* priority queue head */
extern struct prioqrec prioq(void){ return prioq0; }


static struct prioqrec* prioqrec_free = NULL;

static struct prioqrec*
more_prioqrec()
{
  unsigned long k;
  struct prioqrec* bulk =
    (struct prioqrec*) calloc(PRIOQRECBULK, sizeof(struct prioqrec));
  for( k = PRIOQRECBULK-1; k > 0; k-- ){
    bulk[k-1].next = &bulk[k];
  }
  assert( k==0 );

  bulk[PRIOQRECBULK-1].next = 0;
  return bulk;
}

static struct predicate wait_prio_preds[MAXSTDARGS+1];
static struct predicate wait_penum_preds[MAXSTDARGS+1]; /* for inter nodes */

static void
reinitiate_prioq(void)
{
  declare_globals;
  static struct prioqrec* pq;
  static struct prioqrec tail_sentinel = { 0, -1, 0 };
  q* allocp = heapp();

  new_prioqrec(pq);

  /* The tail of the lowest priority level is top level succeed goal. */
  pq->next = &tail_sentinel;
  pq->prio = 0;

#ifdef SHM
  if( my_node == MASTER ){
    allocp[0] = (q) &goal_queue_tail;
    allocp[1] = (q) &mastersucceed_pred;
    pq->q = (struct goalrec*) allocp;
  }else{
    allocp[0] = (q) &goal_queue_tail;
    allocp[1] = (q) &childsucceed_pred;
    pq->q = (struct goalrec*) allocp;
  }
  allocp += 2;
#else
  allocp[0] = (q) &goal_queue_tail;
  allocp[1] = (q) &topsucceed_pred;
  pq->q = (struct goalrec*) allocp;
  allocp += 2;
#endif

  prioq0.next = pq;

  set_heapp(allocp);
}

extern void
initiate_prioq(void)
{
  int k;

  /* Prepare for creating goals that wait for priority value */
  for( k=0; k<=MAXSTDARGS; k++ ){
    wait_prio_preds[k].func = wait_prio_routine;
    wait_prio_preds[k].pred = k;
    wait_prio_preds[k].arity = k+3;
  }

  /* Prepare for creating goals that wait for node value */
  for( k=0; k<=MAXSTDARGS; k++ ){
    wait_penum_preds[k].func = wait_penum_routine;
    wait_penum_preds[k].pred = k;
    wait_penum_preds[k].arity = k+2;
  }

  reinitiate_prioq();
}

extern struct goalrec*
enqueue_goal(qp, prio, gp, glbl)
  struct goalrec* qp;
  long prio;
  struct goalrec* gp;
  struct global_variables* glbl;
{
  if( prio<0 ) prio = 0;
  if( current_prio() == prio ){
    gp->next = qp;
    qp = gp;
  }else{
    if( trace_flag ){
      struct enqueue_trace_rec* tr;
      tr = (struct enqueue_trace_rec*)
	malloc_check(sizeof(struct enqueue_trace_rec));
      tr->next = trace_enqueued_goals;
      tr->g = gp;
      tr->prio = prio;
      trace_enqueued_goals = tr;
    }else
    {
      struct prioqrec* pq = &prioq0;
      while( pq->next->prio > prio ){
	pq = pq->next;
      }
      if( pq->next->prio == prio ){
	/* there already are some active goals with the same priority */
	pq = pq->next;
	gp->next = pq->q;
	pq->q = gp;
      }else{
	/* there are no active goals with the same priority */
	/* must allocate a new prioqrec entry */
	struct prioqrec* newpq;
	new_prioqrec(newpq);
	newpq->next = pq->next;
	pq->next = newpq;
	newpq->prio = prio;
	newpq->q = gp;
	gp->next = &goal_queue_tail;
      }

      /* interrupt for higher priority goal */
      if( current_prio() < prio ){
	set_higher_priority_goal();
	heaplimit = 0;
      }
    }
  }
  return qp;
}

static void
priority_type_error(gp)
  struct goalrec* gp;
{
  fatal("Non-integer priority specified");
}

extern struct goalrec*
enqueue_after_waiting(qp, prio, gp, is_relative)
  struct goalrec* qp;
  q prio;
  struct goalrec* gp;
  int is_relative;
{
  declare_globals;
  struct goalrec* ng;
  q* allocp = heapp();
  int k;
 again:
  if( isint(prio) ){
    return
      enqueue_goal(qp,
		   (is_relative ? current_prio()-intval(prio) : intval(prio)),
		   gp, glbl);
  }
  if( !isref(prio) ) priority_type_error(qp);
  {
    q value = derefone(prio);
    if( value != prio && (!isref(value) || derefone(value) != prio) ){
      prio = value;
      goto again;
    }
  }
  heapalloc(ng, gp->pred->arity+(2+3), (struct goalrec*));
  for( k=0; k < gp->pred->arity; k++ ){
    ng->args[k] = gp->args[k];
  }
  assert(k == gp->pred->arity);

  ng->args[k] = makecons(gp->pred);
  ng->args[k+1] = prio;
  ng->args[k+2] = (is_relative ? makeint(1) : makeint(0));
  ng->pred = &wait_prio_preds[gp->pred->arity];
  add_resumed_goal(ng);
  heaplimit = 0;
  return qp;
}

extern module
wait_prio_routine(glbl, qp, allocp, toppred)
  struct global_variables* glbl;
  struct goalrec* qp;
  q* allocp;
  const struct predicate* toppred;
{
  int arity = toppred->pred;
  q prio = qp->args[arity+1];
  int is_relative = intval(qp->args[arity+2]);

  qp->pred = (struct predicate*) consp(qp->args[arity]);
  set_heapp(allocp);
  qp = enqueue_after_waiting(qp->next, prio, qp, is_relative);
  set_heapp(allocp);
  current_queue = qp;
  return (module) qp->pred->func;
}

extern struct goalrec*
enqueue_throw_goal(penum, gp, qp)
  q penum;
  struct goalrec* gp;
  struct goalrec* qp;
{
  declare_globals;
  struct goalrec* ng;
  q* allocp = heapp();
  int k;
 again:
  if( isint(penum) ){
    int pn = intval(penum);
    if( pn != my_node ){
#ifdef DIST
      return send_throw_goal(intval(penum), gp, qp);
#else
# ifdef SHM
      return throw_goal_routine(intval(penum), qp, gp);
# else
      return enqueue_goal(qp, current_prio(), gp, glbl);
# endif
#endif
    }else return enqueue_goal(qp, current_prio(), gp, glbl);
  }
  if( !isref(penum) ) fatal("inter cluster pragma error");
  {
    q value = derefone(penum);
    if( value != penum && (!isref(value) || derefone(value) != penum) ){
      penum = value;
      goto again;
    }
  }
  heapalloc(ng, gp->pred->arity+(2+2), (struct goalrec*));
  for( k=0; k < gp->pred->arity; k++ ){
    ng->args[k] = gp->args[k];
  }
  assert(k == gp->pred->arity);

  ng->args[k] = makecons(gp->pred);
  ng->args[k+1] = penum;
  ng->pred = &wait_penum_preds[gp->pred->arity];
  add_resumed_goal(ng);
  heaplimit = 0;
  return qp;
}

extern module
wait_penum_routine(glbl, qp, allocp, toppred)
  struct global_variables* glbl;
  struct goalrec* qp;
  q* allocp;
  const struct predicate* toppred;
{
  int arity = toppred->pred;
  q penum = qp->args[arity+1];

  qp->pred = (struct predicate*) consp(qp->args[arity]);
  set_heapp(allocp);
  qp = enqueue_throw_goal(penum, qp, qp->next);
  set_heapp(allocp);
  current_queue = qp;
  return (module) qp->pred->func;
}

extern struct goalrec*
get_top_priority_queue()
{
  declare_globals;
  struct goalrec* newqp;
  struct prioqrec* newprioq;

  newqp = prioq0.next->q;
  set_current_prio(prioq0.next->prio);
  newprioq = prioq0.next->next;
  free_prioqrec(prioq0.next);
  prioq0.next = newprioq;
  return newqp;
}

extern void
put_priority_queue(qp, prio)
  struct goalrec* qp;
  long prio;
{
  declare_globals;
  struct prioqrec* pq = &prioq0;

  while( pq->next->prio >= prio ){
    pq = pq->next;
  }
  if( pq->next->prio == prio ){
    /* there are some active goals with the same priority */
    pq = pq->next;
    qp->next = pq->q;
    pq->q = qp;
  }else{
    /* there are no goals with the same priority */
    struct prioqrec* newpq;
    new_prioqrec(newpq);
    newpq->next = pq->next;
    pq->next = newpq;
    newpq->prio = prio;
    newpq->q = qp;
  }
}

#ifdef SHM
static void
put_priority_goal(qp, prio)
  struct goalrec* qp;
  long prio;
{
  declare_globals;
  struct prioqrec* pq = &prioq0;

  while( pq->next->prio > prio ){
    pq = pq->next;
  }
  if( pq->next->prio == prio ){
    /* there are some active goals with the same priority */
    pq = pq->next;
    qp->next = pq->q;
    pq->q = qp;
  }else{
    /* there are no goals with the same priority */
    struct prioqrec* newpq;
    new_prioqrec(newpq);
    newpq->next = pq->next;
    pq->next = newpq;
    newpq->prio = prio;
    newpq->q = qp;
    qp->next = &goal_queue_tail;
  }
}
#endif

/*
  queue_empty:

  Dummy module to be called when the queue of the current priority
  becomes empty.  The call is automatic by having a dummy goal
  "goal_queue_tail" as the common tail of queues, instead of NULL.
*/

static module
queue_empty(glbl, qp, allocp, fg, toppred)
  struct global_variables* glbl;
  struct goalrec* qp;
  q* allocp;
  struct goalrec* fg;
  const struct predicate* toppred;
{
#ifdef SHM
  if( ext_queued ){
    ext_queued = 0;
    qp = get_invoked_goals(&goal_queue_tail);
    set_heapp(allocp);
    current_queue = qp;
    return (module) qp->pred->func;
  }
#endif
  qp = get_top_priority_queue();
  set_heapp(allocp);
  current_queue = qp;
  return (module) qp->pred->func;
}

#ifdef SHM
extern struct goalrec*
get_invoked_goals(qp)
  struct goalrec* qp;
{
  declare_globals;
  int key = p_key(my_node);

  struct ex_goalrec *sqp, *eqp;

  s_lock(key);
  sqp = ex_qp->next;
  if( sqp == NULL ){ s_unlock(key); return qp; }
  eqp = sqp;
  while( eqp->next != NULL ){ eqp = eqp->next; }
  ex_qp = eqp;
  s_unlock(key);

  put_priority_queue(qp, current_prio());
  eqp = sqp;
  for(;;){
    put_priority_goal(&eqp->goal, eqp->prio);
    if( eqp == ex_qp )  break;
    eqp = eqp->next;
  }

  return get_top_priority_queue();
}


static module
mastersucceed(glbl, qp, allocp, toppred)
  struct global_variables* glbl;
  struct goalrec* qp;
  q* allocp;
  const struct predicate* toppred;
{
  int first, i;

  /* printf("Wait master.\n"); */

 WLoop:
  if( !(ex_qp->next) ){
    set_heapp(allocp);
    klic_interrupt(qp);
    if( current_queue != qp ){
      *cur_status = 1;
      qp = current_queue;
      toppred = qp->pred;
      return (module) toppred->func;
    }

    {
      struct global_variables* tglbl;
      for( i=1; i<total_node; i++ ){
	tglbl = get_otherPE_glbl(i);
	if( top_shm_qp[i] != tglbl->par.aux.shm.ex_qp0 ) goto WLoop;
	if( pe_status[i] ) goto WLoop;
      }
      if( ex_qp->next ) goto WLoop;
      for( i=0; i<total_node; i++ ){ s_lock(p_key(i)); }
      *cur_status = 0;
      for( i=0; i<total_node; i++ ){
	tglbl = get_otherPE_glbl(i);
	if( ( top_shm_qp[i] == tglbl->par.aux.shm.ex_qp0 ) &&
	    !pe_status[i] )  continue;
	for( i=0; i<total_node; i++ ){ s_unlock(p_key(i)); }
	goto WLoop;
      }
      for( i=0; i<total_node; i++ ){  /* One more */
	tglbl = get_otherPE_glbl(i);
	if( ( top_shm_qp[i] == tglbl->par.aux.shm.ex_qp0 ) &&
	    !pe_status[i] )  continue;
	for( i=0; i<total_node; i++ ){ s_unlock(p_key(i)); }
	goto WLoop;
      }
      for( i=total_node-1; i>=0; i-- ){  /* One more */
	tglbl = get_otherPE_glbl(i);
	if( ( top_shm_qp[i] == tglbl->par.aux.shm.ex_qp0 ) &&
	    !pe_status[i] ) continue;
	for( i=0; i<total_node; i++ ){ s_unlock(p_key(i)); }
	goto WLoop;
      }
    }

    for( i=0; i<total_node; i++ ){ s_unlock(p_key(i)); }

    if( suspensions() != resumes() ){
      fatalf(" %d perpetual suspending goals.", suspensions() - resumes());
    }
    {
      struct ex_goalrec* eqp;
      struct goalrec* qp;

      for( i=1; i<total_node; i++ ){
	eqp = (struct ex_goalrec*) galloc(4);
	eqp->prio = HIGHESTPRIO-1;
	qp = &eqp->goal;
	qp->next = 0;
	qp->pred = &topsucceed_pred;
	shm_goal_stack(eqp, i);
      }
      longjmp(klic_topmost, 0);
    }
  }
  qp = get_invoked_goals(qp);
  {
    module func;
    toppred = qp->pred;
    func = (module) toppred->func;
    set_heapp(allocp);
    current_queue = qp;
    return func;
  }
}


static module
childsucceed(glbl, qp, allocp, toppred)
  struct global_variables* glbl;
  struct goalrec* qp;
  q* allocp;
  const struct predicate* toppred;
{
  while( !(ex_qp->next) ){
    set_heapp(allocp);
    klic_interrupt(qp);
    if( current_queue != qp ){
      *cur_status = 1;
      qp = current_queue;
      toppred = qp->pred;
      return (module) toppred->func;
    }
    *cur_status = 0;
  }

  *cur_status = 1;
  klic_barrier();

  qp = get_invoked_goals(qp);
  /* dump_queue(qp); */

  {
    module func;
    toppred = qp->pred;
    func = (module) toppred->func;
    set_heapp(allocp);
    current_queue = qp;
    return func;
  }
}
#endif  /* SHM */
