/* ---------------------------------------------------------- 
%   (C)1993,1994,1995 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */

#ifndef _KLIC_STRUCT_H_
#define _KLIC_STRUCT_H_

/*
  struct.h

  Definition of data structures for KL1/C system

  fix for DEC alpha
*/

#include <klic/basic.h>  /* Extern */
#include <klic/param.h>

/****************************************
  UNIVERSAL ONE-WORD DATA

  All the one-word data is represented as type "q".
  This pointer type will always be used after type conversion.
  Using "struct gazonk *" type is for reducing confusion.
****************************************/

typedef struct gazonk* q;

/****************************************
  POINTER TAGS

  [Pointer Tags]
  Lower 2 bits of pointers are used as tags in the following way.

     ...00: Variable Reference
     ...01: Pointer to a Cons Cell
     ...10: Atomic (integer, symbolic atom, etc)
     ...11: Pointer to a Functor Structure

  Variables without suspended goals are represented as
    self-referencing variables.
  Variables _with_ suspended goals are represented as
    a two-word pointer loop,
    the second word being the suspension record structure.

  [Tags for Atomic Data]
     ..0010: Integer (integers have 28 bits including sign)
     ..0110: Symbolic Atom
     ..1010: Reserved
     ..1110: Reserved
****************************************/

#define PTAGMASK	0x3
#define	VARREF		0x0
#define CONS		0x1
#define ATOMIC		0x2
#define FUNCTOR		0x3

#define STRUCT		0x1
#define FUNCTNOTCONS	0x2
#define ATOMNOTREF	0x2

#define ptagof(x)	((unsigned long)(x) & PTAGMASK)

#define isatomic(x)	(ptagof(x) == ATOMIC)
#define iscons(x)	(ptagof(x) == CONS)
#define isfunctor(x)	(ptagof(x) == FUNCTOR)
#define isref(x)	(ptagof(x) == VARREF)

#define isstruct(x)	((unsigned long)(x) & STRUCT)
#define functnotcons(x)	((unsigned long)(x) & FUNCTNOTCONS)
#define atomicnotref(x)	((unsigned long)(x) & ATOMNOTREF)

#define ATAGBITS	4
#define ATAGMASK	0xFL
#define INT		(ATOMIC+0x0)
#define SYM		(ATOMIC+0x4)

#define atagof(x)	((unsigned long)(x) & ATAGMASK)

#define isint(x)	(atagof(x) == INT)
#define issym(x)	(atagof(x) == SYM)

/****************************************
  DATA STRUCTURES

  All the data structures are to be allocated at 4 byte boundaries.
****************************************/

struct functor {
  q functor;			/* principal functor as atomic q object */
  q args[1];			/* arguments */
};

#define functor_of(s)		(functorp(s)->functor)
#define arg(s, k)		(functorp(s)->args[k])

struct cons {
  q cdr, car;
};

#define car_of(x)		(consp(x)->car)
#define cdr_of(x)		(consp(x)->cdr)

/****************************************
  GOAL MANAGEMENT STRUCTURES
****************************************/

/* Predicate structures are constant structures outside of heap */
/* They are pointed to by atomic tags and treated as atomic in GC */

typedef char*(* module)();

struct predicate {
  module(* func)();		/* pointer to function */
  unsigned short int pred;	/* predicate number */
  unsigned short int arity;	/* number of args */
  const struct arginfo* info;	/* argument mode info */
};

/* Argument mode infomation */

struct arginfo {
  int info;
  /* 0: unknown; 1: input; 2: output; -1: stop mark */
};

/* Goal records */

struct goalrec {
  struct goalrec* next;		/* pointer to next goal */
  const struct predicate* pred;	/* predicate descriptor of the goal */
  q args[6];			/* arguments; number 6 is meaningless */
};

/* Priority queue is currently implemented as a linear list of */
/* priority queue records, in descending order of precedence */

struct prioqrec {
  struct prioqrec* next;
  long prio;			/* signed, to use -1 as sentinel */
  struct goalrec* q;
};

/****************************************
  DATA MANIPULATION MACROS
****************************************/

/* Type conversion */

#define refp(x)		((q*) ((unsigned long)(x) - VARREF))
#define consp(x)	((struct cons*) ((unsigned long)(x) - CONS))
#define functorp(x)	((struct functor*) ((unsigned long)(x) - FUNCTOR))

/* Variables */

#define makeref(x)	((q) ((unsigned long)(x) + VARREF))
#define derefone(x)	(*refp(x))

/* Atomic Values */

#define intval(x)	((long)(x) >> ATAGBITS)
#define symval(x)	((unsigned long)(x) >> ATAGBITS)

#define makeint(n)	((q)(((long)(n) << ATAGBITS) + INT))
#define makesym(n)	((q)(((unsigned long)(n) << ATAGBITS) + SYM))

/* for non-heap addr */
#define makeatomic(adr) ((q)((unsigned long)(adr) + ATOMIC))

/* Conses */
#define makecons(x)	((q)((unsigned long)(x) + CONS))

/* Functors */

#define makefunctor(x) 	((q)((unsigned long)(x) + FUNCTOR))

/* Special Atoms */

#define NILATOM		(makesym(0))
#define PERIODATOM	(makesym(1))

/* Global Variables */

#define current_queue	(glbl->current_queue0)
#define heaplimit	(glbl->heaplimit0)
#define interrupt_off   (glbl->interrupt_off0)
#define generic_arg	(glbl->generic_arg0)
#define reasons		(glbl->reasons0)
#define my_node		(glbl->par.my_num0)
#define total_node	(glbl->par.num_pes0)

Extern struct global_variables {
  /* First four items are accessed frequently */
  q* volatile heaplimit0;	/* copy of real_heaplimit or 0 */
  struct goalrec* current_queue0; /* queue for current priority */
  /* The rest are not accessed as often */
  volatile long interrupt_off0;

  struct {
   /* parallel comm Imp. */
    long my_num0;
    long num_pes0;
    union {
      /* shared-memory Imp. */
      struct {
	long queued0;
	struct ex_goalrec* ex_qp0;
	long currid0;
	long oldid0;
	long shm_htop0;
	long shm_hbyte0;
	long dummy[10];
      } shm;
      /* dist-memory Imp. */

    } aux;
  } par;

  q generic_arg0[MAXGENERICARGS]; /* arguments of generic methods */
  q reasons0[MAXSUSPENSION];	/* suspension reasons */
} globals;

Extern struct global_variables* My_glbl  Init(&globals);

#define declare_globals struct global_variables* glbl = My_glbl


/* runtime/alloc.c */
extern q* heapp(void);
extern q* heaptop(void);
extern q* heapbottom(void);
extern q* real_heaplimit(void);
extern q* new_space_top(void);
extern q* old_space_top(void);
extern unsigned long new_space_size(void);
extern unsigned long old_space_size(void);
extern unsigned long real_heapbytesize(void);
extern unsigned long this_more_space(void);
extern void set_heapp(q* p);
extern void set_heaptop(q* p);
extern void set_heapbottom(q* p);
extern void set_real_heaplimit(q* p);
extern void set_new_space_top(q* p);
extern void set_old_space_top(q* p);
extern void set_new_space_size(unsigned long size);
extern void set_old_space_size(unsigned long size);
extern void set_real_heapbytesize(unsigned long size);
extern void add_this_more_space(unsigned long size);
extern void reset_this_more_space(void);

/* runtime/faisus.c */
extern unsigned long suspensions(void);
extern void reset_suspensions(void);

/* runtime/gc.c */
extern q** gcsp(void);
extern q** gcmax(void);
extern struct suspended_goal_rec* suspended_goal_list(void);
extern void set_gcsp(q** pp);
extern void set_suspended_goal_list(struct suspended_goal_rec* p);

/* runtime/intrpt.c */
extern void set_higher_priority_goal(void);
extern void reset_resumed_goals(void);
extern void add_resumed_goal(struct goalrec* goal);

/* runtime/kmain.c */
extern unsigned long heapsize(void);
extern unsigned long maxheapsize(void);
extern unsigned long incrementsize(void);
extern double maxactiveratio(void);
extern int command_argc(void);
extern char* command_arg(int i);
extern unsigned long resumes(void);
extern unsigned long cum_susps(void);
extern unsigned long cum_resumps(void);
extern void double_heapsize(void);
extern void inc_resumes(void);
#if defined(DIST) && defined(USE_REDUCTION_COUNT)
extern int reduction_count(void);
extern void inc_reduction_count(void);
extern void reset_reduction_count(void);
#endif

/* runtime/sched.c */
extern struct prioqrec prioq(void);
extern unsigned long current_prio(void);
extern void set_current_prio(unsigned long prio);

/* runtime/sysc.kl1 */
extern const struct predicate* postmortem_pred(void);
extern q postmortem_args(void);
extern void reset_postmortem(void);


#endif /* _KLIC_STRUCT_H_ */
