/* ---------------------------------------------------------- 
%   (C)1993, 1994, 1995 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
%   (C)1996, 1997, 1998, 1999 Japan Information Processing Development Center
%       (Read COPYRIGHT-JIPDEC for detailed information.)
----------------------------------------------------------- */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <klic/basic.h>
#include <klic/config.h>
#include "klic.h"
#ifdef USELOCKF
#include <fcntl.h>
#endif
#include <assert.h>

#define ATOMNUMBERBASE 2

extern void* malloc();

static char* dbdir = NULL;
static char* initdbdir = NULL;
static int nocfiles = 0;
static int nolink = 0;
static int makelib = 0;
static char* libmade;

struct atomrec {
  struct atomrec* next;
  int id;
  unsigned char* name;
};
static struct atomrec* atomroot;
static int nextatom;

struct functrec {
  struct functrec* next;
  int id;
  struct atomrec* principal_functor;
  int arity;
};
static struct functrec* functroot;
static int nextfunct;

enum mod_or_class { Mod, Class };

struct modrec {
  struct modrec* next;
  unsigned char* name;		/* name of the module */
  enum mod_or_class morc;	/* whether it is a class or a module */
  struct predrec* defined;	/* predicates defined in the module */
				/* sorted by names alphabetically */
  struct refmodrec* refmod;	/* module referenced from the module */
				/* sorted by names alphabetically */
};
static struct modrec* modroot;
static struct modrec currentmod;

struct refmodrec {
  struct refmodrec* next;
  enum mod_or_class morc;
  unsigned char* name;
};

struct predrec {
  struct predrec* next;
  unsigned char* name;
};

static int changes;

static void error_exit(
  char* format,
  char* arg1,
  char* arg2 )
{
  (void) fprintf(stderr, format, arg1, arg2);
  putc('\n', stderr);
  exit(-1);
}

static void usage_error(
  char* command )
{
  error_exit("Usage: %s [-n] [-x dbdir] [-X initdbdir] file ...",
    command, NULL );
}

static struct modrec* find_module(
  unsigned char* name,
  enum mod_or_class morc,
  struct modrec* root )
{
  int name_is_found;

  for( ; root != NULL; root = root->next) {
    name_is_found = (strcmp((char*) name, (char*) root->name) == 0);
    if(root->morc==morc && name_is_found)
      break;
  }
  assert(root==NULL || (root->morc==morc && name_is_found));
  return root;
}

static struct atomrec* enter_atom(
  unsigned char* name )
{
  struct atomrec* a;
  const int len = strlen((char*) name);

  for (a=atomroot; a!=NULL; a=a->next) {
    if (strcmp((char*) name, (char*) a->name) == 0)
      return a;
  }
  assert(a == NULL);

  a = (struct atomrec*) malloc(sizeof(struct atomrec));
  a->name = (unsigned char*) malloc(len + 1);
  a->name = (unsigned char*) strcpy((char*) a->name, (char*) name);
  a->next = atomroot;
  a->id = nextatom;
  nextatom++;
  atomroot = a;
  changes = 1;
  return a;
}

static void enter_functor(
  unsigned char* name )
{
  struct functrec* f;
  unsigned char* p;
  struct atomrec* functor;
  int arity;

  for (p=name+strlen((char*) name)-1; *p!='_'; p--)
    ;
  *p = '\0';
  arity = atoi(p + 1);
  functor = enter_atom(name);
  for (f=functroot; f!=NULL; f=f->next) {
    if (f->principal_functor == functor &&
	f->arity == arity)
      return;
  }
  assert(f == NULL);

  f = (struct functrec*) malloc(sizeof(struct functrec));
  f->principal_functor = functor;
  f->arity = arity;
  f->next = functroot;
  f->id = nextfunct;
  nextfunct++;
  functroot = f;
  changes = 1;
}

/**
 * Compares the newly read module info in currentmod
 * with the original info in klic.db (if existent),
 * and updates the info if any changes are found
 */
static void update_module_info(void)
{
  struct modrec* m;

  for (m=modroot; m!=NULL; m=m->next) {
    if (m->morc == currentmod.morc &&
	strcmp((char*) m->name, (char*) currentmod.name) == 0) {
      int rm_is_changed;
      struct refmodrec *rm_old, *rm_new;
      for(rm_old = m->refmod, rm_new = currentmod.refmod;
	  rm_old != NULL && rm_new != NULL;
	  rm_old = rm_old->next, rm_new = rm_new->next) {
	rm_is_changed =
	  (strcmp((char*) rm_old->name, (char*) rm_new->name) != 0);
	if(rm_is_changed)
	  break;
      }
      assert(rm_old==NULL || rm_new==NULL || rm_is_changed);

      if (rm_old==NULL || rm_new==NULL) {
	int pred_is_changed;
	struct predrec *p_old, *p_new;
	for(p_old = m->defined, p_new = currentmod.defined;
	    p_old != NULL && p_new != NULL;
	    p_old = p_old->next, p_new = p_new->next) {
	  pred_is_changed =
	    (strcmp((char*) p_old->name, (char*) p_new->name) != 0);
	  if(pred_is_changed)
	    break;
	}
	assert(p_old==NULL || p_new==NULL || pred_is_changed);
	if (p_old==NULL && p_new==NULL) {
	  return;
	}
      }
      m->refmod = currentmod.refmod;
      m->defined = currentmod.defined;
      return;
    }
  }
  assert(m == NULL);

  m = (struct modrec*) malloc(sizeof(struct modrec));
  *m = currentmod;
  m->next = modroot;
  modroot = m;
  return;
}

static void enter_module(
  unsigned char* name,
  enum mod_or_class morc )
{
  if (currentmod.name != NULL) {
    /* not the first module to be entered */
    update_module_info();
  }
  currentmod.name = (unsigned char*) malloc(strlen((char*) name) + 1);
  currentmod.name = (unsigned char*)
    strcpy((char*) currentmod.name, (char*) name);
  currentmod.morc = morc;
  currentmod.defined = NULL;
  currentmod.refmod = NULL;
}

static void enter_predicate(
  unsigned char* name )
{
  struct predrec *p, **r;
  int name_offset;

  for (p=currentmod.defined; p!=NULL; p=p->next) {
    name_offset = strcmp((char*) name, (char*) p->name);
    if (name_offset == 0) return;
    if (name_offset > 0) break;
  }
  assert(p==NULL || name_offset > 0);

  p = (struct predrec*) malloc(sizeof(struct predrec));
  p->name = (unsigned char*) malloc(strlen((char*) name) + 1);
  p->name = (unsigned char*) strcpy((char*) p->name, (char*) name);
  for(r = &currentmod.defined; *r != NULL; r = &(*r)->next) {
    name_offset = strcmp((char*) name, (char*) ((*r)->name));
    if(name_offset >= 0)
      break;
  }
  assert(*r==NULL || name_offset >= 0);

  p->next = *r;
  *r = p;
}

static void enter_ref_module(
  unsigned char* name,
  enum mod_or_class morc )
{
  struct refmodrec *m, **r;
  int name_offset;

  for (m=currentmod.refmod; m!=NULL; m=m->next) {
    if (morc > m->morc) break;
    if (morc == m->morc) {
      name_offset = strcmp((char*) name,(char*) m->name);
      if (name_offset > 0) break;
      if (name_offset == 0) return;
    }
  }
  assert(m==NULL || morc > m->morc || name_offset > 0);

  m = (struct refmodrec*) malloc(sizeof(struct refmodrec));
  m->morc = morc;
  m->name = (unsigned char*) malloc(strlen((char*) name) + 1);
  m->name = (unsigned char*) strcpy((char*) m->name, (char*) name);
  for(r = &currentmod.refmod; *r != NULL; r = &(*r)->next) {
    name_offset = strcmp((char*) name, (char*) ((*r)->name));
    if(morc >= (*r)->morc || name_offset >= 0)
      break;
  }
  assert(*r==NULL || morc >= (*r)->morc || name_offset >= 0);

  m->next = *r;
  *r = m;
}

static void enter_one_line(
  char* buf,
  char* pathname )
{
  if (buf[0] == '#') {
    return;			/* comment line */
  } else if (strncmp(buf,"atom_", 5) == 0) {
    (void) enter_atom(buf+5);
  } else if (strncmp(buf,"functor_",8)==0) {
    enter_functor(buf+8);
  } else if (strncmp(buf,"predicate_",10)==0) {
    enter_predicate(buf+10);
  } else if (strncmp(buf,"module_",7)==0) {
    enter_module(buf+7, Mod);
  } else if (strncmp(buf,"class_",6)==0) {
    enter_module(buf+6, Class);
  } else if (strncmp(buf,"ref_module_",11)==0) {
    enter_ref_module(buf+11, Mod);
  } else if (strncmp(buf,"ref_class_",10)==0) {
    enter_ref_module(buf+10, Class);
  } else {
    error_exit("Unrecognized line in file %s:\n%s",
	       pathname, buf);
  }
}

static void read_db_file(
  FILE* file,
  char* pathname )
{
  char buf[BUFSIZE];
  while (fgets(buf, BUFSIZE, file) == buf) {
    *strchr(buf, '\n') = '\0';
    enter_one_line(buf, pathname);
  }
  (void) fclose(file);
  if (currentmod.name != NULL) {
    /* not the first module to be entered */
    update_module_info();
    currentmod.name = NULL;
  }
}

static void read_ext_file(
  char* pathname )
{
  FILE* file;

  file = fopen(pathname, "r");
  if (file==NULL)
    error_exit("Cannot open file %s", pathname, NULL);
  read_db_file(file, pathname);
}

static char* make_path(
  char* dir,
  char* name )
{
  char* path;
  if (dir != NULL) {
    path = (char*) malloc(strlen(dir) + 1 + strlen(name) + 1);
    (void) sprintf(path, "%s/%s", dir, name);
  } else {
    path = name;
  }
  return path;
}

static FILE* open_cwd_out(
  char* name )
{
  FILE* file;
  file = fopen(name, "w");
  if (file == NULL) {
    error_exit("Cannot open file %s", name, NULL);
  }
  return file;
}

static FILE* open_db_out(
  char* name )
{
  FILE* file;
  char* path = make_path(dbdir, name);
  file = fopen(path, "w");
  if (file == NULL) {
    error_exit("Cannot open file %s", path, NULL);
  }
  return file;
}

static void reverse_atomrecs(void)
{
  struct atomrec* next = atomroot;
  struct atomrec* x = NULL;
  while (next != NULL) {
    struct atomrec* last = x;
    x = next;
    next = x->next;
    x->next = last;
  }
  atomroot = x;
}

static void reverse_functrecs(void)
{
  struct functrec* next = functroot;
  struct functrec* x = NULL;
  while (next != NULL) {
    struct functrec* last = x;
    x = next;
    next = x->next;
    x->next = last;
  }
  functroot = x;
}

static int hexvalue(
  int c,
  unsigned char* name )
{
  if ('0' <= c && c <= '9') return c - '0';
  if ('A' <= c && c <= 'F') return c - 'A' + 10;
  if ('a' <= c && c <= 'f') return c - 'a' + 10;
  error_exit("Invalid atom name: %s", name, NULL);
}

static void real_atom_name(
  unsigned char* name,
  unsigned char* q )
{
  unsigned char* p = name;
  while (*p != '\0') {
    if (*p == '_') {
      int c;
      p++;
      c = *p;
      if (c == '_') {
	*q = '_';
      } else {
	int n = (hexvalue(c, name) << 4);
	p++;
	c = *p;
	*q = n + hexvalue(c, name);
      }
    } else {
      *q = *p;
    }
    p++; q++;
  }
  assert(*p == '\0');
  *q = '\0';
}

static void real_pred_name(
  unsigned char* name,
  unsigned char* q )
{
  unsigned char* p = name;
  unsigned char* e;
  while (*p != '\0' && (*p != '_' || *(p+1) != 'x'))
    p++;
  assert(*p=='\0' || (*p=='_' && *(p+1)=='x'));

  p += 2;
  e = p + strlen((char*) p);
  do{
    e--;
  }while(*e != '_');
  assert(*e == '_');

  while (p < e) {
    if (*p == '_') {
      int c;
      p++;
      c = *p;
      if (c == '_') {
	*q = '_';
      } else {
	int n = (hexvalue(c, name) << 4);
	p++;
	c = *p;
	*q = n + hexvalue(c, name);
      }
    } else {
      *q = *p;
    }
    p++; q++;
  }
  assert(p >= e);
  *q = '\0';
}

static void print_c_string(
  FILE* file,
  unsigned char* str )
{
  int c;
  (void) fprintf(file, "(unsigned char*)");
  (void) putc('\"', file);
  while ((c = *str) != '\0') {
    if (c=='\\' || c=='"')
      (void) putc('\\', file);
    (void) putc(c, file);
    str++;
  }
  assert(c == '\0');
  (void) putc('\"', file);
}


#define TEMPDBNAME  ".klic.db"

static void
print_atom(FILE* atomh, FILE* atomc, unsigned char* name, int id)
{
  unsigned char realname[BUFSIZE];
  if(! nocfiles){
    fprintf(atomh, "#define atom_%s %dL\n", name, id);

    real_atom_name(name, realname);
    putc('\t', atomc);
    print_c_string(atomc, realname);
    putc(',', atomc);
    putc('\n', atomc);
  }
}

static write_db_files(
  time_t inittime )
{
  FILE *atomh, *atomc, *functh, *functc, *newdb;
  struct atomrec* a;
  struct functrec* f;

  newdb = open_db_out(TEMPDBNAME);
  (void) fprintf(newdb, "#KLIC DB for system @%lu\n",
		(unsigned long)inittime);

  reverse_atomrecs();
  if (!nocfiles) {
    atomh = open_cwd_out("atom.h");
    atomc = open_db_out("atom.c");
    (void) fprintf(atomh,
      "#include <klic/atomstuffs.h>\n\n");
    (void) fprintf(atomc,
      "#include <klic/atomstuffs.h>  /* init_atomname */\n" \
      "#include <klic/basic.h>\n\n" \
      "unsigned char* const init_atomname[] = {\n" );
    print_atom(atomh, atomc, "_5B_5D", 0);  /* [] */
    print_atom(atomh, atomc, "_2E", 1);     /* .  */
  }

  for (a=atomroot; a!=NULL; a=a->next) {
    print_atom(atomh, atomc, a->name, a->id);
    fprintf(newdb, "atom_%s\n", a->name);
  }
  assert(a == NULL);

  if (!nocfiles) {
    (void) fprintf(atomc, "};\n\n" \
		"const unsigned long initial_atoms = %d;\n",
		nextatom );
    (void) fclose(atomh);
    (void) fclose(atomc);
  }

  reverse_functrecs();
  if (!nocfiles) {
    functh = open_cwd_out("funct.h");
    functc = open_db_out("funct.c");
    (void) fprintf(functh,
      "#include <klic/functorstuffs.h>\n\n");
    (void) fprintf(functc,
      "#include <klic/atomstuffs.h>\n" \
      "#include <klic/functorstuffs.h>\n" \
      "#include <klic/basic.h>\n\n" \
      "const unsigned long init_functors[] = {\n" );
  }
  for (f=functroot; f!=NULL; f=f->next) {
    if (!nocfiles) {
      (void) fprintf(functh, "#define functor_%s_%d\t%dL\n",
		    f->principal_functor->name, f->arity,
		    f->id);
      (void) fprintf(functc, "\t%dL,\n",
		    f->principal_functor->id);
    }
    (void) fprintf(newdb, "functor_%s_%d\n",
		  f->principal_functor->name, f->arity);
  }
  assert(f == NULL);
  (void) fclose(newdb);

  if (!nocfiles) {
    (void) fprintf(functc, "};\n\n" \
		"const unsigned long init_arities[] = {\n" );
    for (f=functroot; f!=NULL; f=f->next) {
      (void)fprintf(functc, "\t%dL,\n", f->arity);
    }
    assert(f == NULL);
    (void) fprintf(functc, "};\n\n" \
		"const unsigned long initial_functors = %d;\n",
		nextfunct );
    (void) fclose(functh);
    (void) fclose(functc);
  }

  {
    char* newpath = make_path(dbdir, TEMPDBNAME);
    char* dbpath = make_path(dbdir, DBFILENAME);
    if (rename(newpath, dbpath)) {
      error_exit("Can't rename %s to %s", newpath, dbpath);
    }
  }
}

static void write_predicate_file(void)
{
  FILE* predc;
  struct modrec* m;
  if (!nolink) {
    predc = open_db_out("predicates.c");
    (void) fprintf(predc,
		"#include <klic/basic.h>\n" \
		"#include <klic/predinfo.h>\n\n" );
  }

  if (makelib) {
    char libnamebuf[1024];
    FILE* libdb;
    sprintf(libnamebuf, "lib%s.db", libmade);
    libdb = open_db_out(libnamebuf);
    for (m=modroot; m!=NULL; m=m->next) {
      struct predrec* p;
      struct refmodrec* r;
      (void) fprintf(libdb,
		(m->morc==Mod ? "module_%s\n" : "class_%s\n"),
		m->name );
      for (p=m->defined; p!=NULL; p=p->next) {
	(void) fprintf(libdb, "predicate_%s\n", p->name);
      }
      assert(p == NULL);
      for (r=m->refmod; r!=NULL; r=r->next) {
	(void) fprintf(libdb,
		(r->morc==Mod ? "ref_module_%s\n" : "ref_class_%s\n"),
		r->name );
      }
      assert(r == NULL);
    }
    assert(m == NULL);
    (void) fclose(libdb);
  }
  if (!nocfiles) {
    struct modrec* linkedmod = NULL;
    struct searched_mod {
      unsigned char* name;
      enum mod_or_class morc;
    };
#define Push(nam, mc) do{ smdp->name = (nam); smdp->morc = (mc); smdp++; }while(0)
    struct searched_mod searchmod[4096];
    struct searched_mod* smdp = searchmod;
    Push((unsigned char*) "main", Mod);
    Push((unsigned char*) "generic", Mod);
    Push((unsigned char*) "unify__term__dcode", Mod);
    Push((unsigned char*) "integer__arithmetics", Mod);
    while (smdp != searchmod) {
      struct modrec* searched;
      struct refmodrec* refmods;
      --smdp;
      searched = find_module(smdp->name, smdp->morc, modroot);
      if (searched != NULL) {
	struct modrec* newlinked;
	newlinked = (struct modrec*) malloc(sizeof(struct modrec));
	*newlinked = *searched;
	newlinked->next = linkedmod;
	linkedmod = newlinked;
	for (refmods = searched->refmod;
	     refmods != NULL;
	     refmods = refmods->next) {
	  if (find_module(refmods->name, refmods->morc, linkedmod) == NULL) {
	    struct searched_mod* pp;
	    Push(refmods->name, refmods->morc);
	    for (pp=searchmod;
		 pp->morc != refmods->morc ||
		 strcmp((char*) (pp->name), (char*) refmods->name) != 0;
		 ++pp)
	      ;
	    if (pp != smdp-1) smdp--;
	  }
	}
	assert(refmods == NULL);
      }
    }
    assert(smdp == searchmod);
    if (!nolink) {
      for (m=linkedmod; m!=NULL; m=m->next) {
	if (m->morc == Mod) {
	  struct predrec* p;
	  for (p=m->defined; p!=NULL; p=p->next) {
	    (void) fprintf(predc,
			"extern const struct predicate predicate_%s;\n" \
			"static const struct predinfo predinfo_%s = {\n  ",
			p->name, p->name );
	    {
	      unsigned char realname[BUFSIZE];
	      real_pred_name(p->name, realname);
	      (void) print_c_string(predc, realname);
	    }
	    (void) fprintf(predc, ",\n  &predicate_%s\n};\n\n", p->name);
	  }
	  (void) fprintf(predc,
		"static const struct predinfo* const preds_in_%s[] = {\n",
		m->name );
	  for (p=m->defined; p!=NULL; p=p->next) {
	    (void) fprintf(predc, "  &predinfo_%s,\n", p->name);
	  }
	  assert(p == NULL);
	  (void) fprintf(predc, "  0\n};\n");
	}
      }
      (void) fprintf(predc, "const struct modinfo defined_modules[] = {\n");
      for (m=linkedmod; m!=NULL; m=m->next) {
	if (m->morc == Mod) {
	  (void) fprintf(predc, "  ");
	  {
	    unsigned char realname[BUFSIZE];
	    real_atom_name(m->name, realname);
	    (void) print_c_string(predc, realname);
	  }
	  (void) fprintf(predc, ",\n  preds_in_%s,\n", m->name);
	}
      }
      (void) fprintf(predc, "  0, 0\n};\n");
      (void) fclose(predc);
    }
  }
}

static int from_same_installation(
  FILE* klicdb,
  time_t inittime )
{
  int c;
  time_t dbtime;

  c = fgetc(klicdb);
  if (c != '#') return 0;
  assert(c == '#');

  do {
    c = fgetc(klicdb);
    if (c == '\n' || c == EOF) {
      return 0;
    }
    assert(c != '\n' && c != EOF);
  } while (c != '@');
  assert(c == '@');

  {
    unsigned long temp;
    if (fscanf(klicdb, "%lu", &temp) != 1) return 0;
    dbtime = (time_t) temp;
  }
  if (dbtime != inittime) return 0;
  rewind(klicdb);
  return 1;
}


#define Optarg() \
( argv[optind][charind+1] != '\0' ? \
  argv[optind]+charind+1 : \
  (optind++, argv[optind]) )

static void unlink_file(
  char* name )
{
  int err = unlink(name);
  if (err == 0) return;
  if (errno != ENOENT) {
    error_exit("Can't clear file: %s", name, NULL);
  }
}

extern int main(
  int argc,
  char** argv )
{
  int optind;
  char *dbpath, *initpath, *libdbpath;
  FILE *klicdb, *libdb;
  int n;
  int c;
  struct stat initstat, libdbstat;
#ifdef USELOCKF
  int fd;
  char fdbuf[BUFSIZE];
#endif
  for (optind = 1;
       optind<argc && argv[optind][0]=='-';
       optind++) {
    int charind;
    for (charind = 1;
	 argv[optind][charind] != '\0';
	 charind++) {
      switch (c = argv[optind][charind]) {
      case 'X': initdbdir = Optarg(); goto nextarg;
      case 'x': dbdir = Optarg(); goto nextarg;
      case 'n': nocfiles = 1; nolink = 1; break;
      case 'c': nolink = 1; break;
      case 'l': makelib = 1; libmade = Optarg(); goto nextarg;
      default: usage_error(argv[0]);
      }
    }
    assert(argv[optind][charind] == '\0');
  nextarg: ;
  }
  assert(optind==argc || argv[optind][0] != '-');

  if (initdbdir == NULL) {
    fprintf(stderr, "Initial KLIC database directory not specified\n");
    usage_error(argv[0]);
  }

  atomroot = NULL; nextatom = ATOMNUMBERBASE;
  functroot = NULL; nextfunct = 0;
  currentmod.name = NULL;

  initpath = make_path(initdbdir, "klicdb.init");
  libdbpath = make_path(initdbdir, LIBDBNAME);
  dbpath = make_path(dbdir, DBFILENAME);

  if (stat(initpath, &initstat) != 0) {
    error_exit("Can't access file: %s", initpath, NULL);
  }
  if (!makelib && stat(libdbpath, &libdbstat) != 0) {
    error_exit("Can't access file: %s", libdbpath, NULL);
  }

#ifdef USELOCKF
  strcpy(fdbuf, dbpath);
  strcat(fdbuf, ".lock");
  fd = open(fdbuf, O_RDWR);
  lockf(fd, F_LOCK, 1);
#endif

  if (!makelib) {
    libdb = fopen(libdbpath, "r");
    read_db_file(libdb, libdbpath);
    changes = 0;
  }
  klicdb = fopen(dbpath, "r");
  if (klicdb != NULL &&
      from_same_installation(klicdb, initstat.st_mtime)) {
    read_db_file(klicdb, dbpath);
    changes = 0;
  } else {
    FILE* klicinit = fopen(initpath, "r");
    if (klicinit == NULL) {
      error_exit("Can't open initial database file: %s", initpath, NULL);
    }
    unlink_file("atom.h");
    unlink_file(make_path(dbdir, "atom.c"));
    unlink_file("funct.h");
    unlink_file(make_path(dbdir, "funct.c"));
    unlink_file(make_path(dbdir, "predicates.c"));
    read_db_file(klicinit, initpath);
    changes = 1;
  }

  for (n=optind; n<argc; n++) {
    read_ext_file(argv[n]);
  }
  assert(n == argc);

  {
    struct stat buf;
    if (changes ||
	stat(make_path(dbdir, "klic.db"), &buf) != 0 ||
	(!nocfiles &&
	 (stat("atom.h", &buf) != 0 ||
	  stat("funct.h", &buf) != 0 ||
	  stat(make_path(dbdir, "atom.c"), &buf) != 0 ||
	  stat(make_path(dbdir, "funct.c"), &buf) != 0))) {
      write_db_files(initstat.st_mtime);
    }
  }
  write_predicate_file();

#ifdef USELOCKF
  close(fd);
  lockf(fd, F_ULOCK, 1);
#endif

  return 0;
}
