// Copyright (C) 1997  Toyoda Masashi (toyoda@is.titech.ac.jp)

#include <amulet.h>

#include <fstream.h>
#include <strstream.h>
#include <unistd.h>

#include "ToyWidgets.h"
#include "PovlWidgets.h"
#include "Resources.h"
#include "Module.h"
#include "Process.h"
#include "Pattern.h"
#include "Rule.h"
#include "Atom.h"
#include "Message.h"
#include "Guard.h"
#include "Port.h"
#include "Binder.h"

// ----------------------------------------------------------------------
//	Constants and Declarations
// ----------------------------------------------------------------------

Am_Define_Method_Type(Trans_Method, int, (Am_Object, int, ofstream&));
Am_Define_Method_Type_Impl(Trans_Method);

DEFINE_SLOT(TransMethod);
#define CALL_TRANS(obj, flag, ofile) ((Trans_Method)obj.Get(TransMethod)).Call(obj, flag, ofile)

// ----------------------------------------------------------------------
//		ID Management 
// ----------------------------------------------------------------------

// Port$B$N(BId$B$O(BModel$B$K3JG<$9$k!#(B
// -------------------------------------------------------------------
// |  Old Ports   | Goal Ports                         | Inner Ports |
// |--------------+------------------------------------+-------------|
// | i1, i2,...,i | (i+1, i+2), (i+3, i+4)... (j, j+1) | j+2, j+3,...|
// -------------------------------------------------------------------
//   ^ Pattern$B$N$H$-$O(B     ^ Pattern $B$N:F5"$N$H$-$N(Bnext_id
//  $B$=$N?F$N%4!<%k%]!<%H$N(Bid
//
// InnerName  _i_
// InnerNextName _i__

void assign_inner_names(Am_Object port, int id, Am_Object rule)
{
  char buff[1000];
  ostrstream name(buff, 1000);
  name.seekp(0, ios::beg);

  name << "_" << id << "_" << ends;
  port.Set(InnerName, Next_Name(rule, InnerPortNameList, buff));
  name.seekp(0, ios::beg);
  name << "_" << id << "__" << ends;
  port.Set(InnerNextName, Next_Name(rule, InnerPortNameList, buff));
}

void assign_inner_names_with_orig_name(Am_Object port, int id, Am_Object rule)
{
  char buff[1000];
  ostrstream name(buff, 1000);
  name.seekp(0, ios::beg);

  port.Set(InnerName, port.Get_Object(Model).Get(Name));
  name.seekp(0, ios::beg);
  name << "_" << id << "_" << port.Get_Object(Model).Get(Name) << ends;
  port.Set(InnerNextName, Next_Name(rule, InnerPortNameList, buff));
}

int assign_old_port_id_and_name(Am_Object rule)
{
  int id = 0;

  Am_Value_List ports = rule.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    pt.Set(Id, ++id)
      .Set(NextId, 0);
    assign_inner_names(pt, id, rule);
  }
  return id;
}

void assign_ids_in_pattern(Am_Object pattern);

int assign_goal_ports_id_and_name(Am_Object rule, int old_port_max)
{
  int id = old_port_max;

  Am_Value_List goals = rule.Get(ProcessList);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object gl = goals.Get();
    Am_Value_List ports = gl.Get(PortList);
    for (ports.Start(); !ports.Last(); ports.Next()) {
      Am_Object pt = ports.Get();
      pt.Set(Id, ++id);
      assign_inner_names(pt, id, rule);
      pt.Set(NextId, ++id);
    }
    if (gl.Is_Instance_Of(PatternProto)) {
      assign_ids_in_pattern(gl);
    }
  }
  return id;
}

void inner_port_search_and_assign(Am_Object port, Am_Object rule, int goal_port_max)
{
  Am_Value_List names = rule.Get(InnerPortNameList);
  Am_Value_List values = port.Get(ValueList);
  for (values.Start(); !values.Last(); values.Next()) {
    Am_Object val = values.Get();
    if (val.Is_Instance_Of(PortProto)) {
      Am_String name = val.Get_Object(Model).Get(Name);
      int id = goal_port_max + String_List_Index(names, name);
      val.Set(Id, id); 
      assign_inner_names_with_orig_name(val, id, rule);
    }
    if (val.Is_Instance_Of(PortProto) || val.Is_Instance_Of(MessageProto))
      inner_port_search_and_assign(val, rule, goal_port_max);
  }
}

void assign_inner_ports_id_and_name(Am_Object rule, int goal_port_max)
{
  // assign to ports in a rule ports
  Am_Value_List ports = rule.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    if (pt.Is_Instance_Of(PortProto)) inner_port_search_and_assign(pt, rule, goal_port_max);
  }
  // assign to ports in process ports
  Am_Value_List procs = rule.Get(ProcessList);
  for (procs.Start(); !procs.Last(); procs.Next()) {
    Am_Object pr = procs.Get();
    if (pr.Is_Instance_Of(ProcessProto)) {
      Am_Value_List ports = pr.Get(PortList);
      for (ports.Start(); !ports.Last(); ports.Next()) {
	Am_Object pt = (Am_Object)ports.Get();
	if (pt.Is_Instance_Of(PortProto)) inner_port_search_and_assign(pt, rule, goal_port_max);
      }
    }
  }
}

void assign_binder_id_and_name(Am_Object rule)
{
  Am_Value_List binders = rule.Get_Part(BindersPart).Get(Am_GRAPHICAL_PARTS);
  for (binders.Start(); !binders.Last(); binders.Next()) {
    Am_Object bd = binders.Get();
    Am_Object src_port = bd.Get(SrcPort);
    if (src_port.Valid()) {
      Am_Object paren = src_port.Get(Parent);
      if (!paren.Is_Instance_Of(HoleProto)) {
	bd.Set(Id, src_port.Get(Id))
	  .Set(InnerName, (Am_String)src_port.Get(InnerName));
      }
    }
  }
}

// Assign All ID in a rule
// -----------------------
void assign_ids_in_pattern(Am_Object pattern)
{
  // old port$B$K$O$9$G$K(Bgoal port$B$H$7$F$N(BId$B$,IU$1$i$l$F$$$k!#(B
  // goal port$B$H$7$F$N(Bid$B$N(Bmax$B$O(B?
  int max_id = 0;
  Am_Value_List ports = pattern.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    int id = pt.Get(Id);
    if (id > max_id) max_id = id + 1; // goal port$B$O(BId$B$r(B2$B8D>CHq$9$k!#(B
  }
  max_id = assign_goal_ports_id_and_name(pattern, max_id);
  assign_inner_ports_id_and_name(pattern, max_id);
  assign_binder_id_and_name(pattern);
}

void assign_ids_in_rule(Am_Object rule)
{
  int id_count = 0;
  id_count = assign_old_port_id_and_name(rule);
  id_count = assign_goal_ports_id_and_name(rule, id_count);
  assign_inner_ports_id_and_name(rule, id_count);
  assign_binder_id_and_name(rule);
}


// Assign All ID in a module
// -------------------------
void Assign_All_Ids(Am_Object module)
{
  Am_Value_List procs = module.Get(ProcessList);
  for (procs.Start(); !procs.Last(); procs.Next()) {
    Am_Object proc = procs.Get();
    Am_Value_List rules = proc.Get(RuleList);
    int id = 0;
    for (rules.Start(); !rules.Last(); rules.Next(), ++id) {
      Am_Object r = rules.Get();
      r.Set(Id, id);
      assign_ids_in_rule(r);
    }
  }
  Am_Object main_rule = module.Get_Part(ContentsPart).Get(MainRule);
  main_rule.Set(Id, 0);
  assign_ids_in_rule(main_rule);
}

// ----------------------------------------------------------------------
//		Utilities
// ----------------------------------------------------------------------

// Usable macros
// -------------
#define Mode_Str(mode)	((mode == PortInput) ? " +" : " -")
#define Type_Str(type)	((type == PortSingleton) ? " single" : " stream")
#define Dir_Str(dir)	((dir == PortLeft) ? " left" : " right")
#define Open_Str(open)	((open == PortOpen) ? "open" : "close")

// Sorting port list
// -----------------
// $B%*%j%8%J%k$O%b!<%I!&%?%$%W$bHf3SBP>H$@$C$?$,!"L>A0$@$1$K$9$k!#(B
bool is_before_port(Am_Object port1, Am_Object port2)
{
  //  int mode1 = port1.Get_Object(Model).Get(Mode);
  //  int mode2 = port2.Get_Object(Model).Get(Mode);
  //  if (mode1 == PortInput && mode2 == PortOutput) return true;
  //  if (mode1 == PortOutput && mode2 == PortInput) return false;

  //  int type1 = port1.Get_Object(Model).Get(Type);
  //  int type2 = port2.Get_Object(Model).Get(Type);
  //  if (type1 == PortStream && type2 == PortSingleton) return true;
  //  if (type1 == PortSingleton && type2 == PortStream) return false;

  Am_String name1 = (Am_String)port1.Get_Object(Model).Get(Name);
  Am_String name2 = (Am_String)port2.Get_Object(Model).Get(Name);
  if (strcmp(name1, name2) < 0) return true;
  else return false;
}

Am_Value_List sort_ports(Am_Value_List ports)
{
  Am_Value_List new_ports = Am_Value_List();
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    for (new_ports.Start(); ; new_ports.Next()) {
      if (new_ports.Last()) {
	new_ports.Add(port);
	break;
      }
      if (is_before_port(port, new_ports.Get())) {
	new_ports.Insert(port, Am_BEFORE, false);
	break;
      }
    }
  }
  return new_ports;
}

// Format coodinations and sizes
// -----------------------------
int format_point_size(Am_Object obj, ofstream& ofile)
{
  int x = (int)origX(obj);
  int y = (int)origY(obj);
  int w = origW(obj);
  int h = origH(obj);
  Am_Object conts = obj.Get_Part(ContentsPart);
  if (conts.Is_Instance_Of(AlignZoomGroup)) {
    w = Get_Required_Width(conts);
    h = Get_Required_Height(conts);
  }
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  if (w < 10) w = 10;
  if (h < 10) h = 10;
  ofile << " point " << x << " " << y
	<< " size "  << w << " " << h;
  return 1;
}

int format_pointx2_size(Am_Object obj, ofstream& ofile)
{
  int x = origX(obj) * 3 / 2;
  int y = origY(obj) * 3 / 2;
  int w = origW(obj);
  int h = origH(obj);
  Am_Object conts = obj.Get_Part(ContentsPart);
  if (conts.Is_Instance_Of(AlignZoomGroup)) {
    w = Get_Required_Width(conts);
    h = Get_Required_Height(conts);
  }
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  if (w < 10) w = 10;
  if (h < 10) h = 10;
  ofile << " point " << x << " " << y
	<< " size "  << w << " " << h;
  return 1;
}

int format_point_sizex2(Am_Object obj, ofstream& ofile)
{
  int x = origX(obj);
  int y = origY(obj);
  int w = origW(obj) * 2;
  int h = origH(obj) * 2;
  Am_Object conts = obj.Get_Part(ContentsPart);
  if (conts.Is_Instance_Of(AlignZoomGroup)) {
    w = Get_Required_Width(conts);
    h = Get_Required_Height(conts);
  }
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  if (w < 10) w = 10;
  if (h < 10) h = 10;
  ofile << " point " << x << " " << y
	<< " size "  << w << " " << h;
  return 1;
}

int format_only_size(Am_Object obj, ofstream& ofile)
{
  int w = origW(obj);
  int h = origH(obj);
  Am_Object conts = obj.Get_Part(ContentsPart);
  if (conts.Is_Instance_Of(AlignZoomGroup)) {
    w = Get_Required_Width(conts);
    h = Get_Required_Height(conts);
  }
  if (w < 10) w = 10;
  if (h < 10) h = 10;
  ofile << " point " << 0 << " " << 0
	<< " size "  << w << " " << h;
  return 1;
}

// ----------------------------------------------------------------------
//	Main Translation Functions	
// ----------------------------------------------------------------------

#define HEAD_PHASE       1
#define GOAL_PHASE       2
#define LINK_PHASE       3
#define UNIFY_PHASE      4
#define TERM_HEAD_PHASE  5
#define TERM_UNIFY_PHASE 6
#define REC_GOAL_PHASE   7
#define TRANS_BODY_PHASE 8

// Translate Module
// ----------------
Am_Object find_module(Am_Object top_module, Am_String name)
{
  Am_Value_List module_list = top_module.Get(ModuleList);
  Am_Object module = Object_List_Get_By_Model_Key(module_list, Name, name);
  if (!module.Valid()) {
    // try to load module...
    return 0;
  }
  return module;
}

int translate_process(Am_Object process,ofstream& ofile,Am_Value_List& depend);
int translate_patterns(Am_Object module,ofstream& ofile,Am_Value_List& depend);
int translate_patterns_with_layout(Am_Object module,ofstream& ofile,Am_Value_List& depend);

int translate_dependent_module(Am_Object top_module, Am_Value_List& depend, ofstream& ofile)
{
  Am_Value_List translated = Am_Value_List();
  Am_Value_List extra_modules = Am_Value_List();
  
  while(depend.Start(), !depend.Empty()) {
    for (depend.Start(); !depend.Last(); depend.Next()) {
      Am_String module_name = depend.Get();
      if (String_List_Index(translated, module_name) == 0) {
	Am_Object module = find_module(top_module, module_name);
	if (!module.Valid()) return 0;

	Assign_All_Ids(module);
	Am_String module_name = module.MGet(Name);
	ofile << ":- module " << module_name << "." << endl << endl;
	//	translate_patterns(module, ofile, depend);
	translate_patterns_with_layout(module, ofile, depend);
	Am_Value_List procs = module.Get(ProcessList);
	for (procs.Start(); !procs.Last(); procs.Next()) {
	  translate_process(procs.Get(), ofile, extra_modules);
	}
	translated.Add(module_name);
      }
    }
    depend = extra_modules;
    extra_modules = Am_Value_List();
  }
}


int translate_rule(Am_Object rule, ofstream& ofile, Am_Value_List& depend);

int translate_module(Am_Object top_module, Am_Object module, ofstream& ofile)
{
  Am_Value_List depend = Am_Value_List();

  ofile << "/* start of module */" << endl;
  ofile << ":- module main." << endl;
  ofile << "% define 'main normal point 0 0 size 60 60" << endl;
  ofile << "% begin" << endl;
  Am_Object main_rule = module.Get_Part(ContentsPart).Get(MainRule);
  translate_rule(main_rule, ofile, depend);
  ofile << "c_main(N)." << endl;
  ofile << "% end" << endl;
  ofile << "/* end of module */" << endl;

  cout << depend << endl;

  // load and translate dependent modules
  // ====================================
  translate_dependent_module(top_module, depend, ofile);
  return 1;
}

// Translate Process header
// ------------------------
int translate_port_in_proc(Am_Object port, ofstream& ofile)
{
  Am_String port_name = port.MGet(Name);
  char* port_mode = Mode_Str((int)port.Get(Mode));
  char* port_type = Type_Str((int)port.MGet(Type));
  char* port_dir  = Dir_Str((int)port.Get(Dir));

  ofile << "%	port " << port_name << port_mode << port_type << port_dir;
  format_point_size(port, ofile);
  ofile << endl;
  return 1;
}

int translate_process(Am_Object process, ofstream& ofile, Am_Value_List& depend)
{
  Am_String proc_name = process.MGet(Name);
  char* proc_type = " normal";

  // translate spec
  ofile << "% define '" << proc_name << proc_type; 
  format_point_size(process, ofile);
  ofile << endl;
  Am_Value_List ports = sort_ports(process.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = ports.Get();
    translate_port_in_proc(port, ofile);
  }
  ofile << "% begin" << endl;
  /* translate rules */
  Am_Value_List rules = process.Get(RuleList);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = rules.Get();
    translate_rule(rule, ofile, depend);
  }
  ofile << "c_" << proc_name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}


// Translate Arguments
// -------------------
// args$B$OI,MW$J$b$N$@$1$"$i$+$8$aH4$-=P$7$?%j%9%H(B
int translate_arguments(Am_Value_List ports, int flag, ofstream& ofile)
{
  int first = true;
  for (ports.Start(); !ports.Last(); ports.Next(),first = false) {
    if (first) ofile << "( ";
    else       ofile << ", ";

    Am_Object pt = ports.Get();
    if (CALL_TRANS(pt, flag, ofile) == 0) {
      cerr << "translate_arguments(" << ports << ", " << flag << ") at: "<< pt << endl;
      return 0;
    }
    format_point_size(pt, ofile);
    ofile << endl;
  }
  if (!first) ofile << ") ";
  return 1;
}

// Translate Rule
// --------------

// Head part ----------
// $B%X%C%IIt$G$O(Bold port$B$7$+JQ49$7$J$$!#(Btransition$B%k!<%k$N;~6hJL$9$kI,MW$"$j!#(B
Am_Value_List make_port_list_for_head(Am_Object rule)
{
  Am_Value_List new_ports = Am_Value_List();
  Am_Value_List ports = rule.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    if ((int)pt.Get(Age) == PortOld) new_ports.Add(pt);
  }
  return new_ports;
}

// Guard part ----------
int translate_guard_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List guards = rule.Get(GuardList);
  int first = true;
  for (guards.Start(); !guards.Last(); guards.Next(), first = false) {
    if (!first) ofile << ", ";
    Am_Object gd = guards.Get();
    ofile << (Am_String)gd.MGet(Name);
    ofile << "\t % guard" << endl;
  }
  if (first == true) {
    ofile << "true ";
    ofile << "\t % guard" << endl;
  }
  return 1;
}

// Goal part -----------
int translate_goal(Am_Object goal, ofstream& ofile, Am_Value_List& depend)
{
  Am_String name = goal.MGet(Name);
  int type = goal.Get(Type);

  ofile << ", ";
  if (type == ProcessInterGoal) {
    Am_String module_name = goal.MGet(ModuleName);
    ofile << module_name << ":" << name;
    depend.Add(module_name);
  } else {
    ofile << name;
  }
  Am_Value_List ports = sort_ports((Am_Value_List)goal.Get(PortList));
  translate_arguments(ports, GOAL_PHASE, ofile);
  ofile << "\t % proc '" << name;
  format_pointx2_size(goal, ofile);
  ofile << endl;
  return 1;
}

bool is_prior_goal(Am_Object g1, Am_Object g2)
{
  int pri1 = g1.Get(Priority);
  int pri2 = g2.Get(Priority);
  if (pri1 > pri2) return true;
  int y1 = g1.Get(Am_TOP);
  int y2 = g2.Get(Am_TOP);
  if (y1 < y2) return true;
  int x1 = g1.Get(Am_LEFT);
  int x2 = g2.Get(Am_LEFT);
  if (x1 < x2) return true;
  return false;
}

Am_Value_List sort_goals(Am_Value_List goals)
{
  Am_Value_List new_goals = Am_Value_List();
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object g = goals.Get();
    for (new_goals.Start(); ; new_goals.Next()) {
      if (new_goals.Last()) {
	new_goals.Add(g);
	break;
      }
      if (is_prior_goal(g, new_goals.Get())) {
	new_goals.Insert(g, Am_BEFORE, false);
	break;
      }
    }
  }
  return new_goals;
}

int translate_goal_part(Am_Object rule, ofstream& ofile, Am_Value_List& depend)
{
  Am_Value_List goals = sort_goals((Am_Value_List)rule.Get(ProcessList));
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = goals.Get();
    if (translate_goal(goal, ofile, depend) == 0) return 0;
  }
  return 1;
}

// Unify part ----------
int translate_unify_part(Am_Object rule, ofstream& ofile)
{
  // $B%4!<%k$N%]!<%H$r(BLINK_PHASE$B$G7k9g!#(B
  Am_Value_List goals = rule.Get(ProcessList);
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object gl = goals.Get();
    Am_Value_List ports = gl.Get(PortList);
    for (ports.Start(); !ports.Last(); ports.Next()) {
      Am_Object pt = (Am_Object)ports.Get();
      if (CALL_TRANS(pt, LINK_PHASE, ofile) == 0) {
	cerr << "translate_unify_part(" << rule << ") at:" << gl << " port:"<< pt << endl;
	return 0;
      }
    }
  }
  // $B%k!<%k$N%]!<%H$r(BUNIFY_PHASE$B$G7k9g!#(B
  Am_Value_List ports = sort_ports(rule.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    if (CALL_TRANS(pt, UNIFY_PHASE, ofile) == 0) {
      cerr << "translate_unify_part(" << rule << ") at:" << pt << endl;
      return 0;
    }
  }
  return 1;
}

// Transition body part ----------
// singleton port$B$N=PNO!J%k!<%kFb$+$i$_$?!K$K8B$j(Bnew$B$N$_$r07$&!#(B
Am_Value_List make_port_list_for_transition_body(Am_Object rule)
{
  Am_Value_List new_ports = Am_Value_List();
  Am_Value_List ports = rule.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    int type = pt.MGet(Type);
    int age =  pt.Get(Age);
    int mode = pt.Get(Mode);
    if (type == PortStream || age == PortNew ||
	(type == PortSingleton && age == PortOld && mode == PortInput)) new_ports.Add(pt);
  }
  return new_ports;
}

int translate_transition_body(Am_Object rule, ofstream& ofile)
{
  Am_String rule_name = rule.MGet(Name);
  ofile << ", " << rule_name;

  Am_Value_List ports = sort_ports(make_port_list_for_transition_body(rule));
  translate_arguments(ports, TRANS_BODY_PHASE, ofile);

  ofile << "\t % proc '" << rule_name << endl;
  ofile << "% other_part" << endl;

  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    if (true) {// ***** $BMW8+D>$7(B *****
      if (Port_Has_Value(pt) || pt.Get(State) == PortClose) {
	if (CALL_TRANS(pt, UNIFY_PHASE, ofile) == 0) {
	  cerr << "error in port:" << pt << endl;
	  return 0;
	}
      }
    }
  }
  return 1;
}

int translate_rule(Am_Object rule, ofstream& ofile, Am_Value_List& depend)
{
  Am_String rule_name = rule.MGet(Name);
  int rule_id = rule.Get(Id);
  char* rule_type = "";
  if (rule.Is_Instance_Of(TransitionRuleProto)) rule_type = "transition";  else
  if (rule.Is_Instance_Of(NetworkRuleProto) || rule.Is_Instance_Of(PatternProto)) {
    Am_Value_List goals = rule.Get(ProcessList);
    if (goals.Empty()) rule_type = "vanishing";
    else               rule_type = "network";
  } else {
    cerr << "translate_rule(" << rule << "): illegal type" << endl;
    return 0;
  }

  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << rule_type;
  format_point_sizex2(rule, ofile);
  ofile << endl;
  // Head$BIt(B 
  ofile << rule_name;
  Am_Value_List ports = sort_ports(make_port_list_for_head(rule));
  translate_arguments(ports, HEAD_PHASE, ofile);
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(rule, ofile);
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  if (rule.Is_Instance_Of(NetworkRuleProto) || rule.Is_Instance_Of(PatternProto)) {
    translate_goal_part(rule, ofile, depend);
    ofile << "% other_part" << endl;
    translate_unify_part(rule, ofile);
  } else if (rule.Is_Instance_Of(TransitionRuleProto)) {
    translate_transition_body(rule, ofile);
  }
  ofile << "." << endl;
  return 1;
}


// Translate Ports
// ---------------

int format_inner_name(Am_Object port, ofstream& ofile)
{
  ofile << port.Get(InnerName);
  return 1;
}

int format_next_name(Am_Object port, ofstream& ofile)
{
  ofile << port.Get(InnerNextName);
  return 1;
}

int format_port_info(Am_Object port, ofstream& ofile)
{
  ofile << "\t % port " << port.MGet(Name) << " " << port.Get(Id)
	<< Mode_Str(port.Get(Mode)) << Type_Str(port.MGet(Type));
  return 1;
}

int format_port_next_info(Am_Object port, ofstream& ofile)
{
  ofile << "\t % port " << port.MGet(Name) << " " << port.Get(NextId)
	<< Mode_Str(port.Get(Mode)) << Type_Str(port.MGet(Type));
  return 1;
}

int format_port_attached_info(Am_Object port, ofstream& ofile)
{
  ofile << " port " << port.MGet(Name) << " "
	<< Mode_Str(port.Get(Mode)) << Type_Str(port.MGet(Type));
  return 1;
}

int format_singleton(Am_Object port, int flag, ofstream& ofile)
{
  Am_Value_List values = port.Get(ValueList);  values.Start();
  if (values.Empty()) {
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
  } else {
    Am_Object value = values.Get();
    if (CALL_TRANS(value, flag, ofile) == 0) {
      cerr << "format_singleton(" << port << ", " << flag << ") at: " << value << endl;
      return 0;
    }
    format_port_attached_info(port, ofile);
  }
  return 1;
}

int format_stream(Am_Object port, int flag, ofstream& ofile)
{
  Am_Value_List values = port.Get(ValueList);  values.Start();
  if (values.Empty() && port.Get(State) == PortOpen) {
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  } else {
    ofile << "[ ";
    int first = true;
    for (values.Start(); !values.Last(); values.Next(), first = false) {
      if (!first) ofile << ", ";
      Am_Object value = values.Get();
      if (CALL_TRANS(value, flag, ofile) == 0) {
	cerr << "format_stream(" << port << ", " << flag << ") at: " << value << endl;
	return 0;
      }
      ofile << endl;
    }
    if ((int)port.Get(State) == PortOpen) {
      ofile << "| ";  format_inner_name(port, ofile);
    }
    ofile << " ]";
    format_port_info(port, ofile);
    return 1;
  }
}

// Link Phase ----------

int format_link(Am_Object port, Am_Object binder, ofstream& ofile)
{
  ofile << ", " << port.Get(InnerName) << " = " << binder.Get(InnerName);
  ofile << "\t % link " << port.Get(Id) << " " << binder.Get(Id) << endl;
  return 1;
}

int format_link_custom(Am_String p1, Am_String p2, int id1, int id2, ofstream& ofile)
{
  ofile << ", " << p1 << " = " << p2;
  ofile << "\t % link " << id1 << " " << id2 << endl;
  return 1;
}

int link_output_ports(Am_Object port, ofstream& ofile)
{
  Am_Value_List values = port.Get(ValueList);
  for(values.Start(); !values.Last(); values.Next()) {
    Am_Object val = (Am_Object)values.Get();
    CALL_TRANS(val, LINK_PHASE, ofile);
  }
  return 1;
}

int link_singleton(Am_Object port, ofstream& ofile)
{
  int mode = port.Get(Mode);
  // $B%Q%?!<%s$r%4!<%k$H$7$F07$&$H$-$O%]!<%H$N%b!<%I$r5U$K$9$k!#(B
  if ((int)port.MGet(Special) != PortNormal) {
    cout << "****toggle mode of" << port << port.MGet(Name) << endl;
    mode = 1 - mode;
    cout << " " << 1-mode << " to " << mode << endl;
  }
  
  Am_Value_List binders = port.Get(InputBinders);  binders.Start();
    
  if (mode == PortInput && port.Get(State) == PortOpen &&
      !Port_Has_Value(port) && binders.Empty()) {
    cerr << "Port: " << port << "has no value or binder." << endl;
    return 0;
  }
  // Output port
  if (mode == PortOutput) {
    link_output_ports(port, ofile);
    return 1;
  }
  // Input port
  if (mode == PortInput) {
    if (!binders.Empty()) {
      Am_Object bd = binders.Get();
      format_link(port, bd, ofile);
    } else if (Port_Has_Value(port)) {
      Am_Value_List values = port.Get(ValueList);  values.Start();
      Am_Object val = (Am_Object)values.Get();
      if (CALL_TRANS(val, LINK_PHASE, ofile)) { }
    } else {
      cerr << "Port: " << port << "has neither binder nor value." << endl;
      return 0;
    }
    return 1;
  }
}

int link_input_stream(Am_Object port, ofstream& ofile)
{
  Am_Value_List binders = port.Get(InputBinders);  binders.Start();
  if (!binders.Empty()) {
    if (binders.Length() == 1) {
      Am_Object bd = (Am_Object)binders.Get();
      format_link(port, bd, ofile);
    } else {
      ofile << ", generic:new(merge, {";
      int first = true;
      for (binders.Start(); !binders.Last(); binders.Next(), first = false) {
	if (!first) ofile << ", ";
	Am_Object bd = (Am_Object)binders.Get();
	ofile << bd.Get(InnerName);
      }
      ofile << "}, " << port.Get(InnerName) << ")";
      ofile << "\t % merge ";
      ofile << "into " << port.Get(Id) << " from ";
      for (binders.Start(); !binders.Last(); binders.Next()) {
	Am_Object bd = (Am_Object)binders.Get();
	ofile << bd.Get(Id) << " ";
      }
      ofile << endl;
    }
  }
}



int link_stream_no_toggle(Am_Object port, ofstream& ofile)
{
  int mode = port.Get(Mode);
  Am_Value_List binders = port.Get(InputBinders);  binders.Start();
    
  if (mode == PortInput && port.Get(State) == PortOpen &&
      !Port_Has_Value(port) && binders.Empty()) {
    cerr << "Port: " << port << "has no value or binder." << endl;
    return 0;
  }

  if (mode == PortOutput) {
    link_output_ports(port, ofile);
    return 1;
  }

  link_input_stream(port, ofile);
  link_output_ports(port, ofile);
  return 1;
}

int link_stream(Am_Object port, ofstream& ofile)
{
  int mode = port.Get(Mode);
  // $B%Q%?!<%s$r%4!<%k$H$7$F07$&$H$-$O%]!<%H$N%b!<%I$r5U$K$9$k!#(B
  if ((int)port.MGet(Special) != PortNormal) {
    //    cout << "****toggle mode of" << port << port.MGet(Name);
    mode = 1 - mode;
    //    cout << " " << 1-mode << " to " << mode << endl;
  }
  Am_Value_List binders = port.Get(InputBinders);  binders.Start();
    
  if (mode == PortInput && port.Get(State) == PortOpen &&
      !Port_Has_Value(port) && binders.Empty()) {
    cerr << "Port: " << port << "has no value or binder." << endl;
    return 0;
  }

  if (mode == PortOutput) {
    link_output_ports(port, ofile);
    return 1;
  }

  link_input_stream(port, ofile);
  link_output_ports(port, ofile);
  return 1;
}

// Unify Phase ----------

int trans_body_singleton(Am_Object port, ofstream& ofile)
{
  if (CALL_TRANS(port, GOAL_PHASE, ofile) == 0) {
    cerr << "error in port:" << port << endl;
    return 0;
  }
  return 1;
}

int trans_body_stream(Am_Object port, ofstream& ofile)
{
  int mode = port.Get(Mode);
  if (mode == PortOutput) {
    if ((int)port.Get(State) == PortClose) ofile << "[ ]";
    else format_inner_name(port, ofile);
  } else {
    if (!Port_Has_Value(port)) format_inner_name(port, ofile);
    else ofile << port.Get(InnerNextName);
  }
  format_port_info(port, ofile);
  return 1;
}


int unify_output_ports(Am_Object port, ofstream& ofile)
{
  Am_Value_List values = port.Get(ValueList);
  for(values.Start(); !values.Last(); values.Next()) {
    Am_Object val = values.Get();
    if (CALL_TRANS(val, UNIFY_PHASE, ofile) == 0) {}
  }
  return 1;
}

int unify_singleton(Am_Object port, ofstream& ofile)
{
  // Output ports
  if ((int)port.Get(Mode) == PortOutput) {
    unify_output_ports(port, ofile);
    return 1;
  }
  // Input ports
  Am_Value_List binders = port.Get(InputBinders);  binders.Start();
  if (Port_Has_Value(port)) {
    ofile << ", ";
    Am_Value_List values = port.Get(ValueList);  values.Start();
    Am_Object val = (Am_Object)values.Get();
    CALL_TRANS(val, GOAL_PHASE, ofile);
    ofile << endl << " = " << port.Get(InnerName) << "\t % unify " << port.Get(Id) << endl;
    CALL_TRANS(val, LINK_PHASE, ofile); // $BCf?H$N%j%s%/$rD%$j$K9T$/!%(B
  } else if (!binders.Empty()) {
    Am_Object bd = binders.Get();
    format_link(port, bd, ofile);
  } else {
    cerr << "unify_phase(), port: "<< port <<" has neither binder nor value" << endl;
    return 0;
  }
  return 1;
}

int unify_stream(Am_Object port, ofstream& ofile)
{
  // Output ports
  if ((int)port.Get(Mode) == PortOutput) {
    unify_output_ports(port, ofile);
    return 1;
  }
  // Input ports
  Am_Value_List binders = port.Get(InputBinders);
  if (Port_Has_Value(port) || port.Get(State) == PortClose) {
    ofile << ", [ ";
    int first = true;
    Am_Value_List values = port.Get(ValueList);
    for (values.Start(); !values.Last(); values.Next(), first = false) {
      if (!first) ofile << ", ";
      Am_Object value = (Am_Object)values.Get();
      if (CALL_TRANS(value, GOAL_PHASE, ofile) == 0) {
	cerr <<"unify_phase(): error in port"<< port << endl;
	return 0;
      }
      ofile << endl;
    }
    Am_String inner_name = port.Get(InnerName);
    Am_String next_name = port.Get(InnerNextName);
    if (port.Get(State) == PortOpen) ofile << "| " << next_name << " ]";
    else ofile << "]";
    ofile << " = " << inner_name << "\t % unify " << port.Get(Id) << endl;

    port.Set(InnerName, next_name); // $B0J9_$O?7$7$$JQ?t$K%j%s%/$9$k!%(B
    if (CALL_TRANS(port, LINK_PHASE, ofile) == 0) {
      cerr <<"unify_phase(): error in port"<< port << endl;
      return 0;
    }
    port.Set(InnerName, inner_name); // Recover
  } else if (!binders.Empty()) {
    link_stream_no_toggle(port, ofile);
  } else {
    cerr << "unify_phase(), port: "<< port <<" has neither binder nor value" << endl;
    return 0;
  }
  return 1;
}

int check_exception(Am_Object port, int flag, ofstream& ofile)
{
  // Head$BIt=hM}Cf$N(BInput Port
  if ((int)port.Get(Mode) == PortInput && flag == HEAD_PHASE) {
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  }
  // Pattern$B$r%4!<%k$H$7$F=hM}(B
  if (port.Get_Object(Parent).Is_Instance_Of(PatternProto) && flag == GOAL_PHASE) {
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  }
  return 0;
}

Am_Define_Method(Trans_Method, int, Translate_Singleton, (Am_Object port, int flag, ofstream& ofile))
{
  if (check_exception(port, flag, ofile) == 1) return 1;
  switch (flag) {
  case LINK_PHASE:       link_singleton(port, ofile);   break;
  case UNIFY_PHASE:      unify_singleton(port, ofile);  break;
  case TRANS_BODY_PHASE: trans_body_singleton(port, ofile);  break;
  default:               format_singleton(port, flag, ofile);  break;
  }    
  return 1;
}

Am_Define_Method(Trans_Method, int, Translate_Stream, (Am_Object port, int flag, ofstream& ofile))
{
  if (check_exception(port, flag, ofile) == 1) return 1;
  switch (flag) {
  case LINK_PHASE:       link_stream(port, ofile);   break;
  case UNIFY_PHASE:      unify_stream(port, ofile);  break;
  case TRANS_BODY_PHASE: trans_body_stream(port, ofile);  break;
  default:               format_stream(port, flag, ofile);  break;
  }    
  return 1;
}

// Translate Atom
// --------------
Am_Define_Method(Trans_Method, int, Translate_Atom, (Am_Object atom, int flag, ofstream& ofile))
{
  if (flag == UNIFY_PHASE || flag == LINK_PHASE) return 1;
  Am_String name = atom.MGet(Name);
  ofile << "~(" << name << ")" << "\t % atom '" << name << "'"; 
  return 1;
}

// Translate Message
// -----------------
Am_Define_Method(Trans_Method, int, Translate_Message, (Am_Object message, int flag, ofstream& ofile))
{
  Am_String name = message.MGet(Name);
  Am_Value_List values = message.Get(ValueList);

  if (flag == UNIFY_PHASE || flag == LINK_PHASE) {
    for (values.Start(); !values.Last(); values.Next()) {
      Am_Object value = (Am_Object)values.Get();
      if (CALL_TRANS(value,flag,ofile) == 0) {
	cerr << "error in message:" << message << " value: " << value << endl;
	return 0;
      }
    }
    return 1;
  }

  ofile << name;
  translate_arguments(values, flag, ofile);
  ofile << "\t % mesg '" << name;
  return 1;
}

// ----------------------------------------------------------------------
// 		Translate Pattern
// ----------------------------------------------------------------------
int translate_pattern(Am_Object pattern, ofstream& ofile, Am_Value_List& depend);
int translate_repl_pattern(Am_Object pattern, ofstream& ofile, Am_Value_List& depend);
int translate_patterns_iter(Am_Object rule, ofstream& ofile, Am_Value_List& depend);

int translate_pattern_goal(Am_Object goal, ofstream& ofile, Am_Value_List& depend)
{
  if (goal.Is_Instance_Of(ReplPatternProto)) {
    translate_repl_pattern(goal, ofile, depend);
  } else if (goal.Is_Instance_Of(PatternProto)) {
    translate_pattern(goal, ofile, depend);
  } else { return 1; }
  translate_patterns_iter(goal, ofile, depend);
  return 1;
}

int translate_patterns_iter(Am_Object rule, ofstream& ofile, Am_Value_List& depend)
{
  Am_Value_List goals = rule.Get(ProcessList);  goals.Start();
  if (rule.Is_Instance_Of(ReplPatternProto)) {
    Am_Object goal = goals.Get();
    translate_pattern_goal(goal, ofile, depend);
  } else {
    for (goals.Start(); !goals.Last(); goals.Next()) {
      Am_Object goal = goals.Get();
      translate_pattern_goal(goal, ofile, depend);
    }
  }
  return 1;
}

int translate_patterns(Am_Object module, ofstream& ofile, Am_Value_List& depend)
{
  Am_Value_List rules = module.Get(RuleList);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = rules.Get();
    translate_patterns_iter(rule, ofile, depend);
  }
  return 1;
}

// Translate Pattern
// -----------------
int translate_pattern(Am_Object pattern, ofstream& ofile, Am_Value_List& depend)
{
  Am_String name = pattern.MGet(Name);
  /* translate spec */
  ofile << "% define '" << name << " creator ";
  format_point_size(pattern, ofile);
  ofile << endl;
  Am_Value_List ports = sort_ports(pattern.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    translate_port_in_proc(port, ofile);
  }
  ofile << "% begin" << endl;
  /* use translate rules */
  pattern.Set(Id, 0);
  translate_rule(pattern, ofile, depend);

  ofile << "c_" << name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}


// Translate Replication Pattern
// -----------------------------

//
// Utilities
//
Am_Value_List inner_binders(Am_Object port)
{
  Am_Value_List binders = port.Get(Binders);
  if ((int)port.Get_Object(Model).Get(Special) != PortNormal) {
    int mode = port.Get(Mode);
    Am_Slot_Key binders_key = 0;
    if (mode == PortInput)  binders_key = InputBinders;
    if (mode == PortOutput) binders_key = OutputBinders;    
    if (mode == PortUnknown) {
      cerr << "inner_binders(" << port << ") mode unknown" << endl;
      return 0;
    }
    binders = port.Get(binders_key);
  }
  return binders;
}

Am_Value_List outer_binders(Am_Object port)
{
  int mode = port.Get(Mode);
  if ((int)port.MGet(Special) != PortNormal) mode = 1 - mode;
  Am_Slot_Key binders_key = 0;
  if (mode == PortInput)  binders_key = InputBinders;
  if (mode == PortOutput) binders_key = OutputBinders;    
  if (mode == PortUnknown) {
      cerr << "outer_binders(" << port << ") mode unknown" << endl;
      return 0;
    }
  Am_Value_List binders = port.Get(binders_key);
  return binders;
}

Am_Object inner_bound_port(Am_Object port)
{
  Am_Value_List binders = inner_binders(port);  binders.Start();
  if (binders.Empty()) {
    cerr << "inner_bound_port(" << port << ") no binders" << port.Get_Object(Parent).MGet(Name) << endl;
    return 0;
  }
  Am_Object bd = binders.Get();
  Am_Object src = bd.Get(SrcPort);
  Am_Object des = bd.Get(DestPort);
  return (src == port) ? des : src;
}

Am_Object bound_port(Am_Object port)
{
  Am_Value_List binders = outer_binders(port);  binders.Start();
  if (binders.Empty()) {
    //    cerr << "bound_port(" << port << ") no binders" << port.Get_Object(Parent).MGet(Name) << endl;
    return 0;
  }
  Am_Object bd = binders.Get();
  Am_Object src = bd.Get(SrcPort);
  Am_Object des = bd.Get(DestPort);
  return (src == port) ? des : src;
}

Am_Object first_port(Am_Object port)
{
  //                 first_goal          second_goal         third_goal
  // return---[first_src, first_des]---[second_src, _ ]---[ _, third_des]--- port

  cerr << "first_port()" << endl;
  Am_Object third_des = inner_bound_port(port);        if (!third_des.Valid()) return 0;
  Am_Object third_goal = Port_Parent_Proc(third_des);  if (!third_goal.Valid()) return 0;
  cerr << "third_des: " << third_des << " third_goal: " << third_goal << endl;
  Am_Object first_goal = Group_Get_By_Index(third_goal.Get_Owner(), 1); if (!first_goal.Valid()) return 0;
  Am_Value_List third_des_path = Obj_To_Path(third_des, third_goal);
  Am_Object first_des = Path_To_Obj(first_goal, third_des_path);  if (!first_des.Valid()) return 0;
  cerr << "first_des: " << first_des << " first_goal: " << first_goal << endl;
  Am_Object second_src = bound_port(first_des);        if (!second_src.Valid()) return 0;
  Am_Object second_goal = Port_Parent_Proc(second_src);if (!second_goal.Valid())return 0;
  cerr << "second_src: " << second_src << " second_goal: " << second_goal << endl;
  Am_Value_List second_src_pos = Obj_To_Path(second_src, second_goal);
  Am_Object first_src = Path_To_Obj(first_goal, second_src_pos); if (!first_src.Valid()) return 0;
  
  return bound_port(first_src);
}

Am_Object next_link_port(Am_Object port)
{
  // port---[first_src, return]---[srcond_src, _ ]---...
  cerr << "next_link_port()";
  Am_Object first_src = inner_bound_port(port);         if (!first_src.Valid()) return 0;
  cerr << (Am_String)first_src.MGet(Name) << " in ";
  Am_Object first_goal = Port_Parent_Proc(first_src);   if (!first_goal.Valid()) return 0;
  cerr << (Am_String)first_goal.MGet(Name) << endl;
  Am_Value_List first_src_pos = Obj_To_Path(first_src, first_goal);
  Am_Object second_goal = Group_Get_By_Index(first_goal.Get_Owner(), 2); if (!second_goal.Valid())return 0;
  cerr << (Am_String)second_goal.MGet(Name) << " " ;
  Am_Object second_src = Path_To_Obj(second_goal, first_src_pos); if (!second_src.Valid()) return 0;
  cerr << (Am_String)second_src.MGet(Name)  << bound_port(second_src) << endl ;
  return bound_port(second_src);
}

//
// main translate routine
//
char* Seq_Dir_Str[4] = {" rightward ", " downward ", " rightward ", " downward "};

int translate_repl_recursive_rule(Am_Object pattern, ofstream& ofile, Am_Value_List& depend);
int translate_repl_terminate_rule(Am_Object pattern, ofstream& ofile);

int translate_repl_pattern(Am_Object pattern, ofstream& ofile, Am_Value_List& depend)
{
  Am_String name = pattern.MGet(Name);
  int dir = pattern.Get(Dir);
  /* translate spec */
  ofile <<"% define '"<< name <<" creator arrange "<< Seq_Dir_Str[dir];
  format_point_size(pattern, ofile);
  ofile << endl;
  Am_Value_List ports = sort_ports(pattern.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = ports.Get();
    translate_port_in_proc(port, ofile);
  }
  ofile << "% begin" << endl;
  /* translate rules */
  translate_repl_recursive_rule(pattern, ofile, depend);
  translate_repl_terminate_rule(pattern, ofile);
  ofile << "c_" << name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}

// repl recursive rule
//
int recursion_part(Am_Object pattern, ofstream& ofile);
int recursion_unify_part(Am_Object rule, ofstream& ofile);

int translate_repl_recursive_rule(Am_Object pattern, ofstream& ofile, Am_Value_List& depend)
{
  Am_String rule_name = pattern.MGet(Name);
  int rule_id = 0;
  char* rule_type = "network";
  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << rule_type;
  format_point_size(pattern, ofile);
  ofile << endl;
  // Head$BIt(B 
  ofile << rule_name;
  Am_Value_List ports = sort_ports(make_port_list_for_head(pattern));
  translate_arguments(ports, HEAD_PHASE, ofile);
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(pattern, ofile); // $B<h$j$"$($:;D$7$F$*$/!%$I$&$;6u!%(B
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  Am_Value_List goals = pattern.Get(ProcessList); goals.Start();
  if (goals.Empty()) return 0;
  int order = pattern.Get(GenOrder);
  if (order == 0) {
    translate_goal((Am_Object)goals.Get(), ofile, depend); // $B%4!<%k$O$=$N$^$^%H%i%s%9%l!<%H!%(B
    recursion_part(pattern, ofile); // $B:F5"!%(B
  } else {
    recursion_part(pattern, ofile); // $B:F5"!%(B
    translate_goal((Am_Object)goals.Get(), ofile, depend); // $B%4!<%k$O$=$N$^$^%H%i%s%9%l!<%H!%(B
  }
  ofile << "% other_part" << endl;
  recursion_unify_part(pattern, ofile);
  ofile << "." << endl;
  return 1;
}

int recursion_part(Am_Object pattern, ofstream& ofile)
{
  Am_String name = pattern.MGet(Name);
  ofile << ", " << name;
  Am_Value_List ports = sort_ports(pattern.Get(PortList));
  int first = true;
  for (ports.Start(); !ports.Last(); ports.Next(), first = false) {
    if (first) ofile << "( ";
    else       ofile << ", ";
    Am_Object pt = ports.Get();
    if (CALL_TRANS(pt, REC_GOAL_PHASE, ofile) == 0) {
      cerr << "error in pattern: " << pattern << " port: " << pt << endl;
      return 0;
    }
    format_only_size(pt, ofile);
    ofile << endl;
  }
  if (!first) ofile << ")";
  ofile << "\t % proc '" << name;
  Am_Object second_goal = Object_List_Get(pattern.Get(ProcessList), 2);
  format_point_size(second_goal, ofile); // $B#2HVL\$N%4!<%k$HF1$8Bg$-$5!%(B
  ofile << endl;
  return 1;
}

int bound_same_level_goal_index(Am_Object port)
{
  Am_Object dest_port = bound_port(port);	if (!dest_port.Valid()) return 0;
  Am_Object goal = Port_Parent_Proc(dest_port);	if (!goal.Valid()) return 0;
  if (goal.Get_Owner() != Port_Parent_Proc(port).Get_Owner()) return 0;
  return Group_Index(goal);
}

int recursion_unify_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List goals = rule.Get(ProcessList);  goals.Start();
  if (goals.Empty()) return 0;
  Am_Object gl = (Am_Object)goals.Get(); // $B:G=i$N%4!<%k$N$_!%(B
  Am_Value_List ports = gl.Get(PortList);
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    if (bound_same_level_goal_index(pt) != 2) {//$B#2HVL\$K$N$S$k@~$OL5;k!%(B
      if (CALL_TRANS(pt, LINK_PHASE, ofile) == 0) {
        cerr << "error in goal: " << gl << " port: " << pt << endl;
        return 0;
      }
    }
  }
  ports = sort_ports(rule.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = (Am_Object)ports.Get();
    if (CALL_TRANS(pt, UNIFY_PHASE, ofile) == 0) {
      cerr << "error in port:" << pt << endl;
      return 0;
    }
  }
  return 1;
}

// repl termination rule
//
int terminate_unify_part(Am_Object rule, ofstream& ofile);

int translate_repl_terminate_rule(Am_Object pattern, ofstream& ofile)
{
  Am_String rule_name = pattern.MGet(Name);
  int rule_id = 1;
  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << "vanishing";
  format_point_size(pattern, ofile);
  ofile << endl;
  // Head$BIt(B 
  ofile << rule_name;
  Am_Value_List ports = sort_ports(make_port_list_for_head(pattern));
  translate_arguments(ports, TERM_HEAD_PHASE, ofile);
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(pattern, ofile); // $B<h$j$"$($:;D$7$F$*$/!%$I$&$;6u!%(B
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  ofile << "% other_part" << endl;
  terminate_unify_part(pattern, ofile);
  ofile << "." << endl;
  return 1;
}

int terminate_unify_part(Am_Object rule, ofstream& ofile)
{
  Am_Value_List ports = sort_ports(rule.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object pt = ports.Get();
    if (CALL_TRANS(pt, TERM_UNIFY_PHASE, ofile) == 0) {
      cerr << "error in port:" << pt << endl;
      return 0;
    }
  }
  return 1;
}

// ----------------------------------------------------------------------
//		Translate Pattern with Layout Information
// ----------------------------------------------------------------------
int translate_layout_with_name(Am_Object pattern, ofstream& ofile)
{
  Am_Value_List names = pattern.Get(LayoutNameList);
  Am_Value_List sizes = pattern.Get(LayoutSizeList);
  for (names.Start(), sizes.Start(); !names.Last(); names.Next(), sizes.Next()){
    Am_String nm = names.Get();
    Am_Inter_Location sz = sizes.Get();
    bool as_line;
    Am_Object ref;
    int x, y, w, h;
    sz.Get_Location(as_line, ref, x, y, w, h);

    ofile << "%\tlayout \'" << nm << "\' size " << w << " " << h << endl;
  }
}

int translate_layout(Am_Object proc, ofstream& ofile)
{
  Am_Object hole = proc.Get(Hole);
  Am_Value_List sizes = hole.Get(LayoutSizeList);
  for (sizes.Start(); !sizes.Last(); sizes.Next()){
    Am_Inter_Location sz = sizes.Get();
    bool as_line;
    Am_Object ref;
    int x, y, w, h;
    sz.Get_Location(as_line, ref, x, y, w, h);

    ofile << "\t% layout size " << w << " " << h << endl;
  }
}

int translate_pattern_type(Am_Object pattern, ofstream& ofile)
{
  Am_Value_List names = pattern.Get(LayoutNameList);
  if (names.Empty()) {
    ofile << " pattern ";
  } else {
    ofile << " pattern_top ";
  }
}

int translate_pattern_goal_part(Am_Object rule, ofstream& ofile, Am_Value_List& depend)
{
  Am_Value_List goals = sort_goals((Am_Value_List)rule.Get(ProcessList));
  for (goals.Start(); !goals.Last(); goals.Next()) {
    Am_Object goal = goals.Get();
    if (translate_goal(goal, ofile, depend) == 0) return 0;
    translate_layout(goal, ofile);
  }
  return 1;
}

int translate_pattern_rule(Am_Object rule, ofstream& ofile, Am_Value_List& depend)
{
  Am_String rule_name = rule.MGet(Name);
  int rule_id = rule.Get(Id);
  char* rule_type = "";
  if (rule.Is_Instance_Of(TransitionRuleProto)) rule_type = "transition";  else
  if (rule.Is_Instance_Of(NetworkRuleProto) || rule.Is_Instance_Of(PatternProto)) {
    Am_Value_List goals = rule.Get(ProcessList);
    if (goals.Empty()) rule_type = "vanishing";
    else               rule_type = "network";
  } else {
    cerr << "translate_rule(" << rule << "): illegal type" << endl;
    return 0;
  }

  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << rule_type;
  format_point_sizex2(rule, ofile);
  ofile << endl;
  // Head$BIt(B 
  ofile << rule_name;
  Am_Value_List ports = sort_ports(make_port_list_for_head(rule));
  translate_arguments(ports, HEAD_PHASE, ofile);
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(rule, ofile);
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  if (rule.Is_Instance_Of(NetworkRuleProto) || rule.Is_Instance_Of(PatternProto)) {
    translate_pattern_goal_part(rule, ofile, depend);
    ofile << "% other_part" << endl;
    translate_unify_part(rule, ofile);
  } else if (rule.Is_Instance_Of(TransitionRuleProto)) {
    translate_transition_body(rule, ofile);
  }
  ofile << "." << endl;
  return 1;
}


int translate_pattern_with_layout(Am_Object pattern, ofstream& ofile, Am_Value_List& depend)
{
  Am_String name = pattern.MGet(Name);
  /* translate spec */
  ofile << "% define '" << name;
  translate_pattern_type(pattern, ofile);
  format_point_size(pattern, ofile);
  ofile << endl;

  translate_layout_with_name(pattern, ofile);

  Am_Value_List ports = sort_ports(pattern.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = (Am_Object)ports.Get();
    translate_port_in_proc(port, ofile);
  }

  ofile << "% begin" << endl;
  /* use translate rules */
  pattern.Set(Id, 0);
  translate_pattern_rule(pattern, ofile, depend);

  ofile << "c_" << name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}

int translate_layout_repl_recursive_rule(Am_Object pattern, ofstream& ofile, Am_Value_List& depend)
{
  Am_String rule_name = pattern.MGet(Name);
  int rule_id = 0;
  char* rule_type = "network";
  // $B%?%$%H%k(B
  ofile << "% rule " << rule_id << " " << rule_type;
  format_point_size(pattern, ofile);
  ofile << endl;
  // Head$BIt(B 
  ofile << rule_name;
  Am_Value_List ports = sort_ports(make_port_list_for_head(pattern));
  translate_arguments(ports, HEAD_PHASE, ofile);
  // $B%,!<%IIt(B
  ofile << ":- ";
  translate_guard_part(pattern, ofile); // $B<h$j$"$($:;D$7$F$*$/!%$I$&$;6u!%(B
  // $B%\%G%#!<It(B
  ofile << "| c_" << rule_name << "(" << rule_id << ") " << "% rule_id" << endl;
  ofile << "% proc_part" << endl;
  Am_Value_List goals = pattern.Get(ProcessList); goals.Start();
  if (goals.Empty()) return 0;
  int order = pattern.Get(GenOrder);
  Am_Object goal = goals.Get();
  if (order == 0) {
    translate_goal(goal, ofile, depend); // $B%4!<%k$O$=$N$^$^%H%i%s%9%l!<%H!%(B
    translate_layout(goal, ofile);
    recursion_part(pattern, ofile); // $B:F5"!%(B
    translate_layout(goal, ofile);
  } else {
    recursion_part(pattern, ofile); // $B:F5"!%(B
    translate_layout(goal, ofile);
    translate_goal(goal, ofile, depend); // $B%4!<%k$O$=$N$^$^%H%i%s%9%l!<%H!%(B
    translate_layout(goal, ofile);
  }
  ofile << "% other_part" << endl;
  recursion_unify_part(pattern, ofile);
  ofile << "." << endl;
  return 1;
}

int translate_repl_pattern_with_layout(Am_Object pattern, ofstream& ofile, Am_Value_List& depend)
{
  Am_String name = pattern.MGet(Name);
  int dir = pattern.Get(Dir);
  /* translate spec */
  ofile <<"% define '"<< name;
  translate_pattern_type(pattern, ofile);
  ofile << " arrange " << Seq_Dir_Str[dir];
  format_point_size(pattern, ofile);
  ofile << endl;

  translate_layout_with_name(pattern, ofile);

  Am_Value_List ports = sort_ports(pattern.Get(PortList));
  for (ports.Start(); !ports.Last(); ports.Next()) {
    Am_Object port = ports.Get();
    translate_port_in_proc(port, ofile);
  }
  ofile << "% begin" << endl;
  /* translate rules */
  translate_layout_repl_recursive_rule(pattern, ofile, depend);
  translate_repl_terminate_rule(pattern, ofile);
  ofile << "c_" << name << "(N)." << endl;
  ofile << "% end" << endl << endl;
  return 1;
}

int translate_layout_patterns_iter(Am_Object rule, ofstream& ofile, Am_Value_List& depend);

int translate_layout_pattern_goal(Am_Object goal, ofstream& ofile, Am_Value_List& depend)
{
  if (goal.Is_Instance_Of(ReplPatternProto)) {
    translate_repl_pattern_with_layout(goal, ofile, depend);
  } else if (goal.Is_Instance_Of(PatternProto)) {
    translate_pattern_with_layout(goal, ofile, depend);
  } else { return 1; }
  translate_layout_patterns_iter(goal, ofile, depend);
  return 1;
}

int translate_layout_patterns_iter(Am_Object rule, ofstream& ofile, Am_Value_List& depend)
{
  Am_Value_List goals = rule.Get(ProcessList);  goals.Start();
  if (rule.Is_Instance_Of(ReplPatternProto)) {
    Am_Object goal = goals.Get();
    translate_layout_pattern_goal(goal, ofile, depend);
  } else {
    for (goals.Start(); !goals.Last(); goals.Next()) {
      Am_Object goal = goals.Get();
      translate_layout_pattern_goal(goal, ofile, depend);
    }
  }
  return 1;
}

int translate_patterns_with_layout(Am_Object module, ofstream& ofile, Am_Value_List& depend)
{
  Am_Value_List rules = module.Get(RuleList);
  for (rules.Start(); !rules.Last(); rules.Next()) {
    Am_Object rule = rules.Get();
    translate_layout_patterns_iter(rule, ofile, depend);
  }
  return 1;
}

// ----------------------------------------------------------------------
//		Translate Special Ports
// ----------------------------------------------------------------------

int bound_goal_index(Am_Object port)
{
  Am_Object dest_port = inner_bound_port(port);
  if (!dest_port.Valid()) return 0;
  Am_Object goal = Port_Parent_Proc(dest_port);
  if (!goal.Valid()) return 0;
  return Group_Index(goal);
}

Am_Define_Method(Trans_Method, int, Translate_Special, (Am_Object port, int flag, ofstream& ofile))
{
  Am_Object rule = port.Get(Parent);
  if (!rule.Is_Instance_Of(ReplPatternProto)) // $BIaDL$N%Q%?!<%s$N;~$OIaDL$N%]!<%H$H0l=o!#(B
    return Translate_Stream_proc(port, flag, ofile);

  Am_String inner_name = port.Get(InnerName);
  Am_String next_name  = port.Get(InnerNextName);
  int id      = port.Get(Id);
  int next_id = port.Get(NextId);

  Am_Value_List values = port.Get(ValueList); values.Start();
  switch (flag) {
    // stream $B$r2>Dj$7$F$$$k$,$$$$$N$+$J!)(B
    // goal$B$H$7$F(Btranslate$B$9$k>l9g!#(B
  case GOAL_PHASE:
    format_inner_name(port, ofile); format_port_info(port, ofile);
    return 1;
  case REC_GOAL_PHASE:
    if (bound_goal_index(port) == 1) format_next_name(port, ofile);
    else                             format_inner_name(port, ofile);
    format_port_next_info(port, ofile);
    return 1;
  case LINK_PHASE:
    //    if ((int)port.Get(Mode) == PortInput) link_input_stream(port, ofile);
    link_stream(port, ofile);
    return 1;
    // rule $B$H$7$F(Btranslate$B$9$k>l9g(B
  case HEAD_PHASE:
    format_stream(port, HEAD_PHASE, ofile);
    return 1;
  case UNIFY_PHASE:
    // a---p(b,c)---p_iter(a_next,f_next)----f$B$N(B
    if (bound_goal_index(port) == 1) {
      Am_Object pt = inner_bound_port(port); if (!pt.Valid()) return 0;
      int mode = port.Get(Mode);
      // a--b$B$r$D$J$0(B
      if (mode == PortInput)
	format_link_custom(inner_name,(Am_String)pt.Get(InnerName), next_id, (int)pt.Get(Id), ofile);
      // c---a_next $B$r$D$J$0(B
      Am_Object next_port = next_link_port(port);
      if (!next_port.Valid()) return 0;
      Am_String next_port_name = next_port.Get(InnerName);
      int next_port_id = next_port.Get(Id);
      if (mode == PortInput)
	   format_link_custom(next_port_name, next_name, next_port_id, next_id, ofile);
      else format_link_custom(next_name, next_port_name, next_id, next_port_id, ofile);
    }
    return 1;
  case TERM_HEAD_PHASE:
    if (values.Empty()) format_inner_name(port, ofile);
    else                ofile << "[]";
    format_port_info(port, ofile);
    return 1;
  case TERM_UNIFY_PHASE:
    // p_iter(a, b) :- a = b
    if (bound_goal_index(port) == 4) { // 3$BHVL\$O(Bellipsis$B$N<!$J$N$G(Bindex$B$O(B4
      Am_Object first_pt = first_port(port);
      if (!first_pt.Valid()) return 0;
      int first_id = first_pt.Get(Id);
      Am_String first_name = first_pt.Get(InnerName);
      format_link_custom(first_name, inner_name, first_id, id, ofile);
    }
    return 1;
  default:
    cerr << "Translate_Special(): Irregal flag " << flag << endl;
    return 0;
  }
  return 1;
}

Am_Define_Method(Trans_Method, int, Translate_Map_In, (Am_Object port, int flag, ofstream& ofile))
{
  Am_Object rule = port.Get(Parent);
  Am_String name       = port.MGet(Name);
  Am_String inner_name = port.Get(InnerName);
  int id   = port.Get(Id);
  int next_id = port.Get(NextId);
  Am_Value_List values = port.Get(ValueList); values.Start();
  if (values.Empty()) {
    cerr << "trans_map_in_port(): no values " << name <<" in "<< rule.MGet(Name) <<endl;
    return 0;
  }
  switch (flag) {
  case HEAD_PHASE: {
    ofile << "[ ";
    Am_Object value = (Am_Object)values.Get(); // $B6u$G$O$J$$$O$:!%(B
    if (CALL_TRANS(value, flag, ofile) == 0) {
      return 0;
    }
    ofile << endl;
    ofile << "| " << inner_name << " ]";
    format_port_info(port, ofile);
    return 1;
  }
  case GOAL_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case REC_GOAL_PHASE:
    format_inner_name(port, ofile);
    format_port_next_info(port, ofile);
    return 1;
  case LINK_PHASE:
    link_input_stream(port, ofile);
    return 1;
  case UNIFY_PHASE: {
    Am_Object value = (Am_Object)values.Get();
    if (CALL_TRANS(value, LINK_PHASE, ofile) == 0) {
      cerr <<"trans_map_in_port(): error in port"<<value<<endl; return 0;
    }
    format_link_custom(inner_name, inner_name, next_id, id, ofile);
    return 1;
  }
  case TERM_HEAD_PHASE:
    ofile << "[]";
    format_port_info(port, ofile);
    return 1;
  case TERM_UNIFY_PHASE:
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  return 1;
}

Am_Define_Method(Trans_Method, int, Translate_Map_Out, (Am_Object port, int flag, ofstream& ofile))
{
  Am_Object rule = port.Get(Parent);
  Am_String name       = port.MGet(Name);
  Am_String inner_name = port.Get(InnerName);
  Am_String next_name  = port.Get(InnerNextName);
  int id   = port.Get(Id);
  int next_id = port.Get(NextId);

  Am_Value_List values = port.Get(ValueList);  values.Start();
  if (values.Empty()) {
    cerr << "trans_map_out_port(): no values " << name <<" in "<< rule.MGet(Name) <<endl;
    return 0;
  }
  switch (flag) {
  case HEAD_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case GOAL_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case REC_GOAL_PHASE:
    format_next_name(port, ofile);
    format_port_next_info(port, ofile);
    return 1;
  case UNIFY_PHASE: {
    ofile << ", [ ";
    Am_Object value = (Am_Object)values.Get(); // $B6u$G$O$J$$$O$:!%(B
    if (CALL_TRANS(value, GOAL_PHASE, ofile) == 0) {
      return 0;
    }
    ofile << endl;
    ofile << "| " << next_name << " ] = " << inner_name << "\t % unify " << id << endl;
    ofile << ", " << next_name << " = " << next_name
	  << "\t% link " << id << " " << next_id << endl;
    CALL_TRANS(value, LINK_PHASE, ofile); // $B@~$rD%$k!%(B
    return 1;
  }
  case LINK_PHASE:
    return 1;
  case TERM_HEAD_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case TERM_UNIFY_PHASE:
    ofile << ", [] = " << inner_name << "\t % unify " << id << endl;
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  return 1;
}

Am_Define_Method(Trans_Method, int, Translate_Merge, (Am_Object port, int flag, ofstream& ofile))
{
  Am_Object rule = port.Get(Parent);
  Am_String name       = port.MGet(Name);
  Am_String inner_name = port.Get(InnerName);
  Am_String next_name  = port.Get(InnerNextName);
  int id   = port.Get(Id);
  int next_id = port.Get(NextId);

  Am_Value_List binders = port.Get(InputBinders); //$B%Q%?!<%s$N30It=PNO8N!%(B
  binders.Start();
  if (binders.Empty()) {
    cerr << "trans_merge_port(): no binders " << endl;
    return 0;
  }
  switch (flag) {
  case HEAD_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case GOAL_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case REC_GOAL_PHASE:
    format_next_name(port, ofile);
    format_port_next_info(port, ofile);
    return 1;
  case UNIFY_PHASE: {
    Am_Object binder = (Am_Object)binders.Get(); // $B6u$G$O$J$$$O$:!%(B
    ofile << ", generic:new(merge, {" << binder.Get(InnerName);
    ofile << ", " << next_name << "}, " << inner_name << ")";
    ofile << "\t % merge ";
    ofile << "into " << id << " from ";
    ofile << binder.Get(Id) << " " <<  next_id << endl;
    ofile << ", " << next_name << " = " << next_name
	  << "\t% link " << id << " " << next_id << endl;
    return 1;
  }
  case LINK_PHASE:
    return 1;
  case TERM_HEAD_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case TERM_UNIFY_PHASE:
    ofile << ", [] = " << inner_name << "\t % unify " << id << endl;
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  return 1;
}

Am_Define_Method(Trans_Method, int, Translate_Broadcast, (Am_Object port, int flag, ofstream& ofile))
{
  Am_Object rule = port.Get(Parent);
  Am_String name       = port.MGet(Name);
  Am_String inner_name = port.Get(InnerName);
  Am_String next_name  = port.Get(InnerNextName);
  int id   = port.Get(Id);
  int next_id = port.Get(NextId);

  Am_Value_List values = port.Get(ValueList); values.Start();
  switch (flag) {
  case HEAD_PHASE:
    if (!values.Empty() && (int)port.MGet(Type) == PortSingleton) {
      Am_Object val = (Am_Object)values.Get();
      CALL_TRANS(val, HEAD_PHASE, ofile);
      format_port_attached_info(port, ofile);
      return 1;
    } else {
      format_inner_name(port, ofile);
      format_port_info(port, ofile);
      return 1;
    }
  case GOAL_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
    return 1;
  case REC_GOAL_PHASE:
    if (!values.Empty() && (int)port.MGet(Type) == PortSingleton) {
      Am_Object val = (Am_Object)values.Get();
      CALL_TRANS(val, HEAD_PHASE, ofile);
      format_port_attached_info(port, ofile);
      return 1;
    } else {
      format_inner_name(port, ofile);
      format_port_next_info(port, ofile);
      return 1;
    }
  case LINK_PHASE:
    if ((int)port.MGet(Type) == PortSingleton)
      link_singleton(port, ofile);
    else
      link_stream(port, ofile);
    return 1;
  case UNIFY_PHASE:
    if (values.Empty()) {
      format_link_custom(inner_name, inner_name, next_id, id, ofile);
    }
    return 1;
  case TERM_HEAD_PHASE:
    format_inner_name(port, ofile);
    format_port_info(port, ofile);
  case TERM_UNIFY_PHASE:
    return 1;
  default:
    cerr << "trans_map_in_port(): Irregal flag " << flag << endl;
    return 0;
  }
  return 1;
}


// ----------------------------------------------------------------------
//		Execute Module
// ----------------------------------------------------------------------
const char *tracer_name = TRACER;
const char *tracer_program = KLIEG_BINDIR "/" TRACER;

int Execute_Module(Am_Object top_module, Am_Object module)
{
  Assign_All_Ids(module);

  char buff[1000];
  ostrstream name(buff, 1000);
  name.seekp(0, ios::beg);

  name << module.MGet(Name) << ".kl1" << ends;
  ofstream ofile;
  ofile.open(buff, ios::out);
  if (!ofile) {
    cerr << "can't write code to file: " << buff << endl;
    return 0;
  }

  translate_module(top_module, module, ofile);
  ofile.close();

  pid_t child_pid;

  child_pid = fork();
  if (child_pid < 0) {
    perror("Can't execute KLIEG tracer");
    return 0;
  }
  if (child_pid == 0) {
    execl(tracer_program, tracer_name, buff, NULL);
    exit(1);
  }
}

// ----------------------------------------------------------------------
// 		Initialize Routine
// ----------------------------------------------------------------------

void InitializeId(void)
{
  PatternProto.Set(GenOrder, 1);
  ReplPatternProto.Set(GenOrder, 1);
}

void InitializeTranslator(void)
{
  InitializeId();

  AtomProto         .Set(TransMethod, Translate_Atom);
  MessageProto      .Set(TransMethod, Translate_Message);
  InSingletonProto  .Set(TransMethod, Translate_Singleton);
  OutSingletonProto .Set(TransMethod, Translate_Singleton);
  InStreamProto     .Set(TransMethod, Translate_Stream);
  OutStreamProto    .Set(TransMethod, Translate_Stream);
  SpecialSingletonProto.Set(TransMethod, Translate_Singleton);
  SpecialStreamProto   .Set(TransMethod, Translate_Special);
  BroadSingletonProto  .Set(TransMethod, Translate_Broadcast);
  BroadStreamProto     .Set(TransMethod, Translate_Broadcast);
  MergeProto           .Set(TransMethod, Translate_Merge);
  MapInProto           .Set(TransMethod, Translate_Map_In);
  MapOutProto          .Set(TransMethod, Translate_Map_Out);
}
