/*
 * xeplaymidi.cc -- MIDI status display
 *
 * Copyright (C) 1995-1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: xeplaymidi.cc,v 1.14 1998-04-08 14:53:45+09 satoshi Exp $

#define getopt getopt__
#include <stdlib.h>
#undef getopt
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/xpm.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xresource.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Toggle.h>
#ifdef USE_MOTIF
# include <Xm/CascadeB.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#include "getopt.h"
#endif

#include "playmidi.h"
#include "shmem.h"
#include "server.h"
#include "lcdchars.h"
#include "buttons.h"

#define EPLAYMIDI "eplaymidi"	// server name

static Server* server = NULL;
static volatile Register* reg = NULL;

static const char* const version = "(" VERSION ")";

#define CLASS_NAME	"XEplaymidi"
#define RESOURCE_NAME	"xeplaymidi"
#define NAME		"xeplaymidi"	// WM_NAME, WM_ICON_NAME

static const unsigned long SLEEP_TIME_IDLING = 100;	// in milliseconds
static const unsigned long SLEEP_TIME_PLAYING = 10;

static const int MIDI_CHANNEL = 32;
static const int MAX_PART = 32;		// maximum number of parts

static const int VelocityMeterWidth = 3*16-1+4;
static const int MeterWidth = 2+127/4+2;
static const int WheelWidth = 2+127/4+2;

static const int LCDDotWidth = 16;
static const int LCDDotHeight = 6;
static const int LCDLetterHeight = 21;
static const int LCDWidth = LCDDotWidth*16;
static const int LCDHeight = LCDLetterHeight+LCDDotHeight*16;

static const int KB_RANGE = 2+7*7+1;
static const int KeyboardWidth = KB_RANGE*4;
static const int VEL_FADE = 4;

static const char* prgname;

inline void print_version(void);
inline void print_usage(void);

static void cleanup(int sig);

static enum kbMode {kbNormal, kbTrack, kbMemo} kbMode = kbNormal;

extern const char* inst_sc88_pro[128][128];
extern const char* inst_sc88_88[128][128];
extern const char* inst_sc88_55[128][128];
extern const char* drum_sc88_pro[128];
extern const char* drum_sc88_88[128];
extern const char* drum_sc88_55[128];
extern const char* efx_sc88_pro[128][128];
extern const char* inst_xg[128][128];
extern const char* inst_mu100[128][128];
extern const char* sfx_xg[128];
extern const char* drum_sfx_xg[2];
extern const char* drum_xg[128];

extern void init_inst_name(void);

static void ExitAC(Widget w, XEvent* event, String* params, Cardinal* num);
static void KBModeAC(Widget w, XEvent* event, String* params, Cardinal* num);
static void ControlCB(Widget w, const char* command, XtPointer call_data);
static void ControlAC(Widget w, XEvent* event, String* params, Cardinal* num);
static void Control(const char* command);
static void BodyExposeCB(Widget parent, XtPointer body, XtPointer call_data);
static void LCDDisplayExposeCB(Widget parent, XtPointer lcd,
			       XtPointer call_data);
static void FileCB(Widget w, XtPointer index, XtPointer call_data);
static void InstCB(Widget w, XtPointer index, XtPointer call_data);

static void timer(XtPointer p, XtIntervalId* id);

static String fallback_resources[] = {
  // Parts
  "*nparts: 16",
  "*part.order: "
  "  vel,vol,expr,pan,pitch,mod,rev,cho,dly,prg,bnk,map,name,keyboard",
  // Fonts
  "*font:      -adobe-helvetica-medium-r-*-*-14-*-*-*-*-*-iso8859-*",
  "*text_font: -adobe-courier-medium-r-normal--16-110-100-0-m-*-iso8859-*",
  "*l_C1.font: -adobe-helvetica-medium-r-*-*-10-*-*-*-*-*-iso8859-*",
  "*menubar*font: -adobe-helvetica-medium-r-*-*-12-*-*-*-*-*-iso8859-*",
#ifdef KANJI
  "*track.international: True",
  "*track.FontSet: "
  "  -adobe-courier-medium-r-normal--16-110-100-0-m-*-iso8859-1,"
  "  -jis-fixed-medium-r-normal--16-110-100-100-c-*-jisx0208.1983-0,"
  "  -misc-fixed-medium-r-normal--16-110-100-100-c-*-jisx0201.1976-0",
  "*memo.international: True",
  "*memo.FontSet: "
  "  -adobe-courier-medium-r-normal--16-110-100-0-m-*-iso8859-1,"
  "  -jis-fixed-medium-r-normal--16-110-100-100-c-*-jisx0208.1983-0,"
  "  -misc-fixed-medium-r-normal--16-110-100-100-c-*-jisx0201.1976-0",
  "*title.international: True",
  "*title.FontSet: "
  "  -adobe-courier-medium-r-normal--16-110-100-0-m-*-iso8859-1,"
  "  -jis-fixed-medium-r-normal--16-110-100-100-c-*-jisx0208.1983-0,"
  "  -misc-fixed-medium-r-normal--16-110-100-100-c-*-jisx0201.1976-0",
  "*text.international: True",
  "*text.FontSet: "
  "  -adobe-courier-medium-r-normal--16-110-100-0-m-*-iso8859-1,"
  "  -jis-fixed-medium-r-normal--16-110-100-100-c-*-jisx0208.1983-0,"
  "  -misc-fixed-medium-r-normal--16-110-100-100-c-*-jisx0201.1976-0",
#endif
  // Colors
  "*foreground:				black",
  "*background: 			gray80",

  "*illuminatedEdgeColor:		gray90",
  "*shadowedEdgeColor:			gray40",

  "*ch.background:			gray70",
  "*l_master.background:		gray70",

  "*velocity.foreground:		dark orange",
  "*volume.foreground:			forest green",
  "*expression.foreground:		yellow1",
  "*modulation.foreground:		blue",
  "*reverb.foreground:			lavender",
  "*chorus.foreground:			peach puff",
  "*delay.foreground:			misty rose",
  "*panpot.foreground:			orange red",
  "*pitchbender.foreground:		white",
  "*pitchbender.movedForeground:	orange2",
  "*name.capitalForeground:		black",
  "*name.variationForeground:		dark green",
  "*name.drumForeground:		orange red",
  "*keyboard.blackForeground:		gray10",
  "*keyboard.whiteForeground:		floral white",
  "*keyboard.noteonForeground:		pink",
  "*mvolume.foreground:			forest green",
  "*mpanpot.foreground:			orange red",
  // LCD
  CLASS_NAME ".lcd.foreground:		OrangeRed4",
  CLASS_NAME ".lcd.background:		orange red",
  "*lcd.letter.foreground:		OrangeRed4",
  "*lcd.letter.foreground2:		OrangeRed4",
  "*lcd.letter.background:		orange red",
  // Logos
  CLASS_NAME ".gmLogo:"	EPLAYMIDIPATH "/gm.xpm",
  CLASS_NAME ".gsLogo:"	EPLAYMIDIPATH "/gs.xpm",
  CLASS_NAME ".xgLogo:"	EPLAYMIDIPATH "/xg.xpm",
//  "*iconPixmap:"	EPLAYMIDIPATH "/icon.xpm",
  // Labels
  "*l_title.label:		Title: ",
  "*l_status.label:		Status: ",
  "*l_efxtype.label:		EFX: ",

  "*l_ch.labelString:		ch",
  "*l_master.labelString:	M",
  "*l_vel.labelString:		vel",
  "*l_vol.labelString:		vol",
  "*l_expr.labelString:		expr",
  "*l_pan.labelString:		pan",
  "*l_pitch.labelString:	pitch",
  "*l_mod.labelString:		mod",
  "*l_rev.labelString:		rev",
  "*l_cho.labelString:		cho",
  "*l_dly.labelString:		dly",
  "*l_prg.labelString:		prg",
  "*l_bnk.labelString:		bnk",
  "*l_map.labelString:		map",
  "*l_name.labelString:		instrument name",
//"*l_C1.labelString:		C1",
  "*ch.relief:			True",
  "*prg.relief:			True",
  "*bnk.relief:			True",
  "*map.relief:			True",
  "*l_master.relief:		True",
  "*name.relief:		True",
  "*text.relief:		True",

  "*l_ch.width:			20",
  "*l_ch.height:		16",
  "*l_ch.alignment:		ALIGNMENT_CENTER",
  "*l_master.width:		20",
  "*l_master.height:		16",
  "*l_vel.width:		51",
  "*l_vel.height:		16",
  "*l_vel.alignment:		ALIGNMENT_CENTER",
  "*l_vol.width:		35",
  "*l_vol.height:		16",
  "*l_vol.alignment:		ALIGNMENT_CENTER",
  "*l_expr.width:		35",
  "*l_expr.height:		16",
  "*l_expr.alignment:		ALIGNMENT_CENTER",
  "*l_pan.width:		35",
  "*l_pan.height:		16",
  "*l_pan.alignment:		ALIGNMENT_CENTER",
  "*l_pitch.width:		35",
  "*l_pitch.height:		16",
  "*l_pitch.alignment:		ALIGNMENT_CENTER",
  "*l_mod.width:		35",
  "*l_mod.height:		16",
  "*l_mod.alignment:		ALIGNMENT_CENTER",
  "*l_rev.width:		35",
  "*l_rev.height:		16",
  "*l_rev.alignment:		ALIGNMENT_CENTER",
  "*l_cho.width:		35",
  "*l_cho.height:		16",
  "*l_cho.alignment:		ALIGNMENT_CENTER",
  "*l_dly.width:		35",
  "*l_dly.height:		16",
  "*l_dly.alignment:		ALIGNMENT_CENTER",
  "*l_prg.width:		28",
  "*l_prg.height:		16",
  "*l_prg.alignment:		ALIGNMENT_CENTER",
  "*l_bnk.width:		28",
  "*l_bnk.height:		16",
  "*l_bnk.alignment:		ALIGNMENT_CENTER",
  "*l_map.width:		28",
  "*l_map.height:		16",
  "*l_map.alignment:		ALIGNMENT_CENTER",
  "*l_name.width:		116",
  "*l_name.height:		16",
  "*l_name.alignment:		ALIGNMENT_CENTER",
  "*l_C1.width:			16",
  "*l_C1.height:		16",
  "*l_sc88.width:		16",
  "*l_sc88.height:		16",
  "*l_xg.width:			16",
  "*l_xg.height:		16",
  "*ch.width:			20",
  "*ch.height:			16",
  "*prg.width:			28",
  "*prg.height:			16",
  "*bnk.width:			28",
  "*bnk.height:			16",
  "*map.width:			28",
  "*map.height:			16",
  "*name.width:			116",
  "*name.height:		16",
  "*text.width:			240",
  "*text.height:		20",

  //
  "*Paned*showGrip:		False",
  "*Paned*internalBorderWidth:	0",

  "*Label.borderWidth:		0",
  "*Label.internalHeight:	0",
  "*Label.internalWidth:	0",
  "*Label.justify:		Left",

  "*menubar*background:			white",
  "*menubar.FileMenu.label:		File",
  "*menubar.FileMenu*Open.label:	Open",
  "*menubar.FileMenu*Quit.label:	Quit",
  "*menubar.InstMenu.label:		Options",
  "*menubar.InstMenu*sc55.label:	SC-55",
  "*menubar.InstMenu*sc88.label:	SC-88",
  "*menubar.InstMenu*sc88pro.label:	SC-88Pro",
  "*menubar.InstMenu*xg.label:		XG",

  // for motif
//"*menubar*international:		False",
  "*menubar.button_0.labelString:	File",
//"*menubar.FileMenu.mnemonic:		F",
  "*FileMenu.button_0.labelString:	Open",
//"*FileMenu.button_0.mnemonic:		O",
//"*FileMenu.button_0.accelerator:	Alt<Key>O",
//"*FileMenu.button_0.acceleratorText:	Alt-O",
  "*FileMenu.button_1.labelString:	Quit",
//"*FileMenu.button_1.mnemonic:		Q",
//"*FileMenu.button_1.accelerator:	Alt<Key>Q",
//"*FileMenu.button_1.acceleratorText:	Alt-Q",

  "*Form*horizDistance:		0",
  "*Form*vertDistance:		0",

  "*titleForm.borderWidth:	0",
  "*titleForm*top:		ChainTop",
  "*titleForm*bottom:		ChainTop",
  "*titleForm*left:		ChainLeft",
  "*titleForm*right:		ChainLeft",
  "*l_title.left:		ChainLeft",
  "*l_title.right:		ChainRight",
  "*title.left:			ChainLeft",
  "*title.right:		ChainRight",
  "*title.fromHoriz:		l_title",
  "*title.resizable:		True",

  "*body.allowResize:		True",

  "*lcd.resizeToPreferred:	True",

  "*statusForm.borderWidth:	0",
  "*status.fromHoriz:		l_status",
  "*statusForm*top:		ChainBottom",
  "*statusForm*bottom:		ChainBottom",
  "*statusForm*left:		ChainLeft",
  "*statusForm*right:		ChainLeft",
  "*status.resizable:		True",

  "*efxtypeForm.borderWidth:	0",
  "*efxtype.fromHoriz:		l_efxtype",
  "*efxtypeForm*top:		ChainBottom",
  "*efxtypeForm*bottom:		ChainBottom",
  "*efxtypeForm*left:		ChainLeft",
  "*efxtypeForm*right:		ChainLeft",
  "*l_efxtype.resize:		False",
  "*efxtype.resizable:		True",

  "*logos.gsLogo.fromHoriz:	gmLogo",
  "*logos.xgLogo.fromHoriz:	gsLogo",
  "*logos*top:			ChainBottom",
  "*logos*bottom:		ChainBottom",
  "*logos*left:			ChainLeft",
  "*logos*right:		ChainLeft",
  "*logos*resizable:		True",

  "*buttons.orientation:	horizontal",
  "*buttons.resizeToPreferred:	True",
  "*buttons.borderWidth:	0",

  "*Command.horizDistance:	2",
  "*Command.internalHeight:	2",
  "*Command.internalWidth:	2",

  "*Toggle.horizDistance:	2",
  "*Toggle.internalHeight:	2",
  "*Toggle.internalWidth:	2",

  "*paned*translations: #override"
  "			Alt<Key>Q:	Exit()\\n\\"
  "			Alt<Key>2:	KBMode()\\n\\"
  "			Alt<Key>F:	Control(fast)\\n\\"
  "			Alt<Key>N:	Control(normal)\\n\\"
  "			Alt<Key>S:	Control(slow)\\n\\"
  "			Alt<Key>P:	Control(pause)\\n\\"
  "			Alt<Key>C:	Control(continue)\\n\\"
  "			Alt<Key>B:	Control(previous)\\n\\"
  "			Alt<Key>X:	Control(next)\\n\\"
  "			Alt<Key>D:	Control(fade)\\n\\",
  "			Alt<Key>+:	Control(pup)\n\\",
  "			Alt<Key>-:	Control(pdown)\n",
  NULL
};

static XtActionsRec actions[] = {
  {"Exit",    ExitAC},
  {"KBMode",  KBModeAC},
  {"Control", ControlAC},
};


class Body;
class MenuBar;
class TitleLabel;
class LCDDisplay;
class StatusLabel;
class EfxtypeLabel;
class Logo;
class Timer;

/*
 * Main Window
 */
class XWin {
public: // ???
  XtAppContext appcontext;
private:
  Display* d;
  Window w;
  XrmDatabase rdb, rdb2;
  Widget toplevel;
  Widget paned;
  MenuBar* mb;
  TitleLabel* title;
  Body* body;
  Widget paned1, paned2;
  StatusLabel* status;
  EfxtypeLabel* efxtype;
  Timer* timer;
  Logo* gmLogo;
  Logo* gsLogo;
  Logo* xgLogo;
  Widget logos;
  Widget buttons;
  LCDDisplay* lcd;
  Pixmap playPixmap, pausePixmap, stopPixmap, fwrdPixmap;
  Pixmap backPixmap, nextPixmap,  prevPixmap, quitPixmap;
  Widget playButton, pauseButton, stopButton, fwrdButton;
  Widget backButton, nextButton,  prevButton, quitButton;

  char* wmname;

public:
  XWin(int argc, char* argv[]);
  ~XWin();
  const char* getResource(const char* name);
  const char* getResource(const char* name, const char* name2);
  bool readXpm(const char* xpmfile, Pixmap* pixmap, int* width, int* height);
  void changeTitle(const char* name);
  void main(void);
  void redraw(void);

  friend void ControlCB(Widget /*w*/, const char* command, XtPointer);
  friend void timer(XtPointer p, XtIntervalId* id);
};


/*
 *
 */
bool XWin::readXpm(const char* xpmfile, Pixmap* pixmap,
		   int* width, int* height)
{
  Colormap cmap = DefaultColormap(XtDisplay(toplevel), 0);
  XpmAttributes attr;
  attr.valuemask = XpmColormap | XpmSize /*| XpmCloseness*/;
  attr.colormap  = cmap;
  attr.exactColors = False;
//attr.closeness = closeness;
  Pixmap mask;
  int result =
    XpmReadFileToPixmap(XtDisplay(toplevel),
			XRootWindowOfScreen(XtScreen(toplevel)),
			(char*)xpmfile, pixmap, &mask, &attr);
  *width  = attr.width;
  *height = attr.height;
  if(result != XpmSuccess && result != XpmColorError) {
    fprintf(stderr, "XpmReadFileToPixmap: error(%d): %s.\n", result, xpmfile);
    *pixmap = None;
    *width = 1;
    *height = 1;
    return false;
  }
  return true;
}


/*
 * change window name and icon name
 */
void XWin::changeTitle(const char* name)
{
  delete[] wmname;
  if(name == NULL || name[0] == '\0') {
    wmname = NULL;
    XStoreName(XtDisplay(toplevel), XtWindow(toplevel), NAME);
    XSetIconName(XtDisplay(toplevel), XtWindow(toplevel), NAME);
  } else {
    wmname = new char[2+strlen(NAME)+1+strlen(name)+1];
    sprintf(wmname, "%s: %s", NAME, name);
    XStoreName(XtDisplay(toplevel), XtWindow(toplevel), wmname);
    XSetIconName(XtDisplay(toplevel), XtWindow(toplevel), wmname);
  }
}


/*
 * get resource
 */
const char* XWin::getResource(const char* name)
{
  char rname[80];
  char cname[80];
  sprintf(rname, "%s.%s", RESOURCE_NAME, name);
  sprintf(cname, "%s.%s", CLASS_NAME, name);
  // resource name is just lowercased name of its class name.
  int i;
  for(i = 0; rname[i] != '\0'; i++)
    rname[i] = tolower(rname[i]);
  char* type;
  XrmValue value;
  if(XrmGetResource(rdb, rname, cname, &type, &value))
    return value.addr;
  else
    return "";
}


const char* XWin::getResource(const char* name, const char* name2)
{
  char rname[80];
  char cname[80];
  sprintf(rname, "%s.%s.%s", RESOURCE_NAME, name, name2);
  sprintf(cname, "%s.%s.%s", CLASS_NAME, name, name2);
  int i;
  for(i = 0; rname[i] != '\0'; i++)
    rname[i] = tolower(rname[i]);
  char* type;
  XrmValue value;
  if(XrmGetResource(rdb, rname, cname, &type, &value))
    return value.addr;
  else
    return "";
}


/*
 * Base Widget
 */
class BaseWidget {
protected:
  XWin* xwin;
  Widget parent;
  Display* d;
  Window w;
  GC gc;
  int x, y;
  int width, height;

public:
  BaseWidget(XWin* xwin_, Widget parent_, int x_, int y_) :
  xwin(xwin_), parent(parent_), x(x_), y(y_) {
    d = XtDisplay(parent);
    w = XtWindow(parent);
    XGCValues gcv;
    gcv.graphics_exposures = False;
    gc = XCreateGC(d, XRootWindowOfScreen(XtScreen(parent)),
		   GCGraphicsExposures, &gcv);
  }

  virtual ~BaseWidget() {
    XFreeGC(d, gc);
  }

  void getSize(int& width_, int& height_) const {
    width_ = width;
    height_ = height;
  }

  int getTop(void) const {return y;}
  int getBottom(void) const {return y+height;}
  int getLeft(void) const {return x;}
  int getRight(void) const {return x+width;}

  virtual void update(void) {}
  virtual void redraw(void) {}

  /*
   * allocate named color
   */
  unsigned long allocColor(const char* color) const {
    Colormap cmap;
    XColor c0, c1;
    cmap = DefaultColormap(d, 0);
    XAllocNamedColor(d, cmap, color, &c1, &c0);
    return(c1.pixel);
  }
};


/*
 * Velocity Meter
 */
class Velocity : public BaseWidget {
private:
  volatile Register* reg;
  int part;
  unsigned long foreground, background;
  unsigned long illuminatedEdgeColor, shadowedEdgeColor;
  unsigned char cur_vel;
  unsigned char _vel0;

public:
  Velocity(XWin* xwin_, Widget parent_, int x_, int y_,
	   volatile Register* reg_, int part_) :
  BaseWidget(xwin_, parent_, x_, y_), reg(reg_), part(part_), cur_vel(0),
  _vel0(0) {
    width = VelocityMeterWidth;
    height = 16;
    foreground =
      allocColor(xwin->getResource("Velocity", "Foreground"));
    background =
      allocColor(xwin->getResource("Velocity", "Background"));
    illuminatedEdgeColor =
      allocColor(xwin->getResource("Velocity", "illuminatedEdgeColor"));
    shadowedEdgeColor =
      allocColor(xwin->getResource("Velocity", "shadowedEdgeColor"));
    redraw();
  }

  bool draw(unsigned char vel, unsigned char _vel) {
    if(_vel0 != _vel) {
      _vel0 = _vel;
      draw_(vel);
      return true;
    } else
      return false;
  }

  void redraw(void) {
    XSetForeground(d, gc, background);
    XFillRectangle(d, w, gc, x+2, y+2, (127/8)*3+2, 12);

    XSetForeground(d, gc, foreground);
    int i;
    for(i = 0; i < cur_vel/8; i++)
      XFillRectangle(d, w, gc, x+2+i*3, y+2, 2, 12);

    // relief
    XSetForeground(d, gc, shadowedEdgeColor);
    XDrawLine(d, w, gc, x, y, x+2+(127/8)*3+2+1, y);
    XDrawLine(d, w, gc, x, y, x, y+16-1);
    XDrawLine(d, w, gc, x+2, y+16-2, x+2+(127/8)*3+2+1, y+16-2);
    XDrawLine(d, w, gc, x+2+(127/8)*3+2, y+2, x+2+(127/8)*3+2, y+16-2);
    XSetForeground(d, gc, illuminatedEdgeColor);
    XDrawLine(d, w, gc, x+1, y+16-1, x+2+(127/8)*3+2+1, y+16-1);
    XDrawLine(d, w, gc, x+2+(127/8)*3+2+1, y+2, x+2+(127/8)*3+2+1, y+16-2);
    XDrawLine(d, w, gc, x+1, y+1, x+2+(127/8)*3+2+1, y+1);
    XDrawLine(d, w, gc, x+1, y+1, x+1, y+16-2);
  }

  void update(void) {
    if(!draw(reg->vel[part], reg->_vel[part])) {
      if(cur_vel > 0) {
	if(cur_vel > VEL_FADE)
	  draw_(cur_vel-VEL_FADE);
	else
	  cur_vel = 0;
      }
    }
  }

private:
  void draw_(unsigned char vel) {
    if(vel == 0)	// note off
      return;
    unsigned char vel0 = cur_vel/8;
    cur_vel = vel;
    vel /= 8;
    if(vel0 == vel)
      return;
    if(vel0 < vel) {
      XSetForeground(d, gc, foreground);
      int i;
      for(i = vel0; i < vel; i++)
	XFillRectangle(d, w, gc, x+2+i*3, y+2, 2, 12);
    } else {
      XSetForeground(d, gc, background);
      XFillRectangle(d, w, gc, x+2+vel*3, y+2, (vel0-vel)*3-1, 12);
    }
  }
};


/*
 * Generic Meter
 */
class Meter : public BaseWidget {
private:
  unsigned long foreground, background;
  unsigned long illuminatedEdgeColor, shadowedEdgeColor;
  unsigned char cur_val;

public:
  Meter(XWin* xwin_, Widget parent_, const char* name, int x_, int y_,
	unsigned char val = 0) :
  BaseWidget(xwin_, parent_, x_, y_), cur_val(val) {
    width = MeterWidth;
    height = 16;
    foreground = allocColor(xwin->getResource(name, "Foreground"));
    background = allocColor(xwin->getResource(name, "Background"));
    illuminatedEdgeColor =
      allocColor(xwin->getResource(name, "illuminatedEdgeColor"));
    shadowedEdgeColor =
      allocColor(xwin->getResource(name, "shadowedEdgeColor"));
    redraw();
  }

  void draw(unsigned char val) {
    if(cur_val/4 != val/4) {
#if 1
      if(cur_val < val) {
	XSetForeground(d, gc, foreground);
	XFillRectangle(d, w, gc, x+2+cur_val/4, y+2, val/4-cur_val/4, 12);
      } else {
	XSetForeground(d, gc, background);
	XFillRectangle(d, w, gc, x+2+val/4, y+2, cur_val/4-val/4, 12);
      }
#endif
      cur_val = val;
//    redraw();		// new version
    }
  }

  void redraw(void) {
#if 1
    XSetForeground(d, gc, foreground);
    XFillRectangle(d, w, gc, x+2, y+2, cur_val/4, 12);
    XSetForeground(d, gc, background);
    XFillRectangle(d, w, gc, x+2+cur_val/4, y+2, 127/4-cur_val/4, 12);

    XSetForeground(d, gc, shadowedEdgeColor);
    XDrawLine(d, w, gc, x, y, x+2+127/4+1, y);
    XDrawLine(d, w, gc, x, y, x, y+16-1);
    XDrawLine(d, w, gc, x+2, y+16-2, x+2+127/4+1, y+16-2);
    XDrawLine(d, w, gc, x+2+127/4, y+2, x+2+127/4, y+16-2);
    XSetForeground(d, gc, illuminatedEdgeColor);
    XDrawLine(d, w, gc, x+1, y+16-1, x+2+127/4+1, y+16-1);
    XDrawLine(d, w, gc, x+2+127/4+1, y+2, x+2+127/4+1, y+16-2);
    XDrawLine(d, w, gc, x+1, y+1, x+2+127/4+1, y+1);
    XDrawLine(d, w, gc, x+1, y+1, x+1, y+16-2);
#else
    // new version
    XSetForeground(d, gc, foreground);
    XFillRectangle(d, w, gc,
		   x+2, y+2, cur_val/4, 12);
    XSetForeground(d, gc, background);
    if(cur_val/4 > 0)
      XFillRectangle(d, w, gc, x+2+cur_val/4+1, y+2, 127/4-cur_val/4, 12+1);   //
    else
      XFillRectangle(d, w, gc, x+2+cur_val/4, y+2, 127/4-cur_val/4+1, 12+1);   //

    XSetForeground(d, gc, illuminatedEdgeColor);
    XDrawLine(d, w, gc, x, y, x+2+127/4+1, y);
    XDrawLine(d, w, gc, x, y, x, y+16-1);
    XSetForeground(d, gc, shadowedEdgeColor);
    if(cur_val/4 > 0) {
      XDrawLine(d, w, gc, x+2, y+16-2, x+2+cur_val/4, y+16-2);		//
      XDrawLine(d, w, gc, x+2+cur_val/4, y+2, x+2+cur_val/4, y+16-2);	//
    }
    XSetForeground(d, gc, illuminatedEdgeColor);
    XDrawLine(d, w, gc, x+1, y+1, x+2+127/4+1, y+1);
    XDrawLine(d, w, gc, x+1, y+1, x+1, y+16-2);
    XSetForeground(d, gc, shadowedEdgeColor);
    XDrawLine(d, w, gc, x+1, y+16-1, x+2+127/4+1, y+16-1);
    XDrawLine(d, w, gc, x+2+127/4+1, y+2, x+2+127/4+1, y+16-2);
#endif
  }
};


class VolumeMeter : public Meter {
private:
  volatile Register* reg;
  int part;

public:
  VolumeMeter(XWin* xwin_, Widget parent_, int x, int y,
	      volatile Register* reg_, int part_) :
  Meter(xwin_, parent_, "Volume", x, y, reg_->vol[part_]), reg(reg_),
  part(part_) {}

  void update(void) {
    draw(reg->vol[part]);
  }

  void redraw(void) {
    Meter::redraw();
  }
};


class MVolumeMeter : public Meter {
private:
  volatile Register* reg;

public:
  MVolumeMeter(XWin* xwin_, Widget parent_, int x, int y,
	       volatile Register* reg_) :
  Meter(xwin_, parent_, "MVolume", x, y, reg_->mvol), reg(reg_) {}

  void update(void) {
    draw(reg->mvol);
  }

  void redraw(void) {
    Meter::redraw();
  }
};


class ExpressionMeter : public Meter {
private:
  volatile Register* reg;
  int part;

public:
  ExpressionMeter(XWin* xwin_, Widget parent_, int x, int y,
		  volatile Register* reg_, int part_) :
  Meter(xwin_, parent_, "Expression", x, y, reg_->expr[part_]), reg(reg_),
  part(part_) {}

  void update(void) {
    draw(reg->expr[part]);
  }

  void redraw(void) {
    Meter::redraw();
  }
};


class ModulationMeter : public Meter {
private:
  volatile Register* reg;
  int part;

public:
  ModulationMeter(XWin* xwin_, Widget parent_, int x, int y,
		  volatile Register* reg_, int part_) :
  Meter(xwin_, parent_, "Modulation", x, y, reg_->mod[part_]), reg(reg_),
  part(part_) {}

  void update(void) {
    draw(reg->mod[part]);
  }

  void redraw(void) {
    Meter::redraw();
  }
};


class ReverbMeter : public Meter {
private:
  volatile Register* reg;
  int part;

public:
  ReverbMeter(XWin* xwin_, Widget parent_, int x, int y,
	      volatile Register* reg_, int part_) :
  Meter(xwin_, parent_, "Reverb", x, y, reg_->rev[part_]), reg(reg_),
  part(part_) {}

  void update(void) {
    draw(reg->rev[part]);
  }

  void redraw(void) {
    Meter::redraw();
  }
};


class ChorusMeter : public Meter {
private:
  volatile Register* reg;
  int part;

public:
  ChorusMeter(XWin* xwin_, Widget parent_, int x, int y,
	      volatile Register* reg_, int part_) :
  Meter(xwin_, parent_, "Chorus", x, y, reg_->cho[part_]), reg(reg_),
  part(part_) {}

  void update(void) {
    draw(reg->cho[part]);
  }

  void redraw(void) {
    Meter::redraw();
  }
};


class DelayMeter : public Meter {
private:
  volatile Register* reg;
  int part;

public:
  DelayMeter(XWin* xwin_, Widget parent_, int x, int y,
	     volatile Register* reg_, int part_) :
  Meter(xwin_, parent_, "Delay", x, y, reg_->delay[part_]), reg(reg_),
  part(part_) {}

  void update(void) {
    draw(reg->delay[part]);
  }

  void redraw(void) {
    Meter::redraw();
  }
};


/*
 * Generic Wheel
 */
class Wheel : public BaseWidget {
private:
  unsigned long foreground, background;
  unsigned long illuminatedEdgeColor, shadowedEdgeColor;
  unsigned long movedForeground;
  unsigned char cur_val;

public:
  Wheel(XWin* xwin_, Widget parent_, const char* name, int x_, int y_,
	unsigned char val = 0) : BaseWidget(xwin_, parent_, x_, y_),
  cur_val((unsigned char)((double)val/127.0*123.0)) {
    width = WheelWidth;
    height = 16;
    foreground = allocColor(xwin->getResource(name, "Foreground"));
    background = allocColor(xwin->getResource(name, "Background"));
    const char* movedForeground_str =
      xwin->getResource(name, "movedForeground");
    movedForeground = ((!strcmp(movedForeground_str, ""))
		       ? foreground
		       : allocColor(movedForeground_str));
    illuminatedEdgeColor =
      allocColor(xwin->getResource(name, "illuminatedEdgeColor"));
    shadowedEdgeColor =
      allocColor(xwin->getResource(name, "shadowedEdgeColor"));
    redraw();
  }

  void draw(unsigned char val) {
//  if(val >= 127/4*4) val = 123;	// rewrite!!!
    val = (unsigned char)((double)val/127.0*123.0);
    const unsigned char center = (unsigned char)(64.0/127.0*123.0);
    if(cur_val/4 != val/4 ||
       cur_val != val && (cur_val == /*64*/center || val == /*64*/center)) {
      XSetForeground(d, gc, background);
      XFillRectangle(d, w, gc, x+2+cur_val/4, y+2, 1, 12);
      cur_val = val;
      XSetForeground(d, gc,
		     (cur_val == /*64*/center) ? foreground : movedForeground);
      XFillRectangle(d, w, gc, x+2+cur_val/4, y+2, 1, 12);
    }
  }

  void redraw(void) {
    const unsigned char center = (unsigned char)(64.0/127.0*123.0);
    XSetForeground(d, gc, background);
    XFillRectangle(d, w, gc, x+2, y+2, 127/4, 12);
    XSetForeground(d, gc,
		   (cur_val == /*64*/center) ? foreground : movedForeground);
    XFillRectangle(d, w, gc, x+2+cur_val/4, y+2, 1, 12);

    XSetForeground(d, gc, shadowedEdgeColor);
    XDrawLine(d, w, gc, x, y, x+2+127/4+1, y);
    XDrawLine(d, w, gc, x, y, x, y+16-1);
    XDrawLine(d, w, gc, x+2, y+16-2, x+2+127/4+1, y+16-2);
    XDrawLine(d, w, gc, x+2+127/4, y+2, x+2+127/4, y+16-2);
    XSetForeground(d, gc, illuminatedEdgeColor);
    XDrawLine(d, w, gc, x+1, y+16-1, x+2+127/4+1, y+16-1);
    XDrawLine(d, w, gc, x+2+127/4+1, y+2, x+2+127/4+1, y+16-2);
    XDrawLine(d, w, gc, x+1, y+1, x+2+127/4+1, y+1);
    XDrawLine(d, w, gc, x+1, y+1, x+1, y+16-2);
  }
};


class PanpotWheel : public Wheel {
private:
  volatile Register* reg;
  int part;

public:
  PanpotWheel(XWin* xwin_, Widget parent_, int x, int y,
	      volatile Register* reg_, int part_) :
  Wheel(xwin_, parent_, "Panpot", x, y, reg_->pan[part_]), reg(reg_),
  part(part_) {}

  void update(void) {
    draw(reg->pan[part]);
  }

  void redraw(void) {
    Wheel::redraw();
  }
};


class MPanpotWheel : public Wheel {
private:
  volatile Register* reg;

public:
  MPanpotWheel(XWin* xwin_, Widget parent_, int x, int y,
	       volatile Register* reg_) :
  Wheel(xwin_, parent_, "Panpot", x, y, reg_->mpan), reg(reg_) {}

  void update(void) {
    draw(reg->mpan);
  }

  void redraw(void) {
    Wheel::redraw();
  }
};


class BenderWheel : public Wheel {
private:
  volatile Register* reg;
  int part;

public:
  BenderWheel(XWin* xwin_, Widget parent_, int x, int y,
	      volatile Register* reg_, int part_) :
  Wheel(xwin_, parent_, "Pitchbender", x, y, reg_->bend[part_]/**/&0x7f/**/),
  reg(reg_), part(part_) {}

  void update(void) {
    draw(reg->bend[part]/**/&0x7f/**/);		// ???
  }

  void redraw(void) {
    Wheel::redraw();
  }
};


/*
 * Keyboard
 */
class Keyboard : public BaseWidget {
private:
  volatile Register* reg;
  int part;
  unsigned long blackForeground;
  unsigned long whiteForeground;
  unsigned long noteonForeground;
  unsigned long background;
  unsigned char begin, end;
  unsigned char key[128];
  static const unsigned char table[12];

public:
  Keyboard(XWin* xwin_, Widget parent_, int x_, int y_,
	   volatile Register* reg_, int part_) :
  BaseWidget(xwin_, parent_, x_, y_), reg(reg_), part(part_), begin(21),
  end(108) {
    width = KeyboardWidth;
    height = 16;
    blackForeground =
      allocColor(xwin->getResource("Keyboard", "blackForeground"));
    whiteForeground =
      allocColor(xwin->getResource("Keyboard", "whiteForeground"));
    noteonForeground =
      allocColor(xwin->getResource("Keyboard", "noteonForeground"));
    background =
      allocColor(xwin->getResource("Keyboard", "Background"));
    int note;
    for(note = 0; note < 128; note++)
      key[note] = 0;
    redraw();
  }

  void update(void) {
    int note;
    for(note = 0; note < 128; note++)
      draw(note, reg->kb[part][note]);
  }

  void redraw(void) {
    int note;
    for(note = begin; note <= end; note++)
      draw_(note, key[note]);
  }

  void draw(unsigned char note, unsigned char on) {
    if(key[note] != on) {
      key[note] = on;
      if(begin <= note && note <= end)
	draw_(note, on);
    }
  }

private:
  void draw_(unsigned char note, unsigned char on) {
    unsigned char n = note%12;
    unsigned char wn = (note-begin)/12*7;
    int i;
    for(i = 0; i < (note-begin)%12; i++)
      wn += table[(begin+i)%12];
    switch(n) {
    case 1: case 3: case 6: case 8: case 10:
      XSetForeground(d, gc, on ? noteonForeground : blackForeground);
      XFillRectangle(d, w, gc, x+wn*4-2, y+1, 3, 8);
      break;
    case 0: case 5:
      XSetForeground(d, gc, on ? noteonForeground : whiteForeground);
      XFillRectangle(d, w, gc, x+wn*4,   y+1, 2, 15);
      XFillRectangle(d, w, gc, x+wn*4+2, y+9, 1, 7);
      break;
    case 2: case 7: case 9:
      XSetForeground(d, gc, on ? noteonForeground : whiteForeground);
      XFillRectangle(d, w, gc, x+wn*4,   y+9, 1, 7);
      XFillRectangle(d, w, gc, x+wn*4+1, y+1, 1, 15);
      XFillRectangle(d, w, gc, x+wn*4+2, y+9, 1, 7);
      break;
    case 4: case 11:
      XSetForeground(d, gc, on ? noteonForeground : whiteForeground);
      XFillRectangle(d, w, gc, x+wn*4,   y+9, 1, 7);
      XFillRectangle(d, w, gc, x+wn*4+1, y+1, 2, 15);
      break;
    }
  }
};

const unsigned char Keyboard::table[12] =
  {1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 1};


/*
 * LCD Display
 */
class LCDDisplay {
private:
  XWin* xwin;
  Widget parent;
  Display* d;
  Window w;
  GC gc;

  Widget lcd;

  volatile Register* reg;
  int nparts;
  unsigned long foreground, background;
  unsigned long letterForeground, letterForeground2, letterBackground;
  Pixmap chars[96];		// LCD characters

  char letter[32];
  unsigned char lcd_dd[1+10+1][16][16];	// page 0x0b is used for a level meter
  unsigned char page;
  unsigned char time;

  char patch_name[16];		// patch name
  char cur_letter[16];		// displaying letter
  int letter_length;		// valid length of the letter
  int display_length;		// length of the letters displayed
  bool flag;			// 
  long limit_letter;		// time limit
  long limit_dot;		// time limit

  // for a level meter
  unsigned char cur_vel[MAX_PART];
  unsigned char v[MAX_PART], v0[MAX_PART];

public:
  LCDDisplay(XWin* xwin_, Widget parent_, int nparts_,
	     volatile Register* reg_) :
  xwin(xwin_), parent(parent_), reg(reg_), nparts(nparts_), page(1),
  time(6), flag(false), limit_letter(0), limit_dot(0) {
    lcd = XtVaCreateManagedWidget("lcd", coreWidgetClass, parent,
				  XtNwidth,       LCDWidth,
				  XtNheight,      LCDHeight,
				  XtNborderWidth, 0,
				  NULL);
    XtVaSetValues(lcd, XtNmax, LCDWidth, NULL);
    XtVaSetValues(lcd, XtNmin, LCDWidth, NULL);
  }

  void initialize(void) {
    d = XtDisplay(lcd);
    w = XtWindow(lcd);
    XGCValues gcv;
    gcv.graphics_exposures = False;
    gc = XCreateGC(d, XRootWindowOfScreen(XtScreen(parent)),
		   GCGraphicsExposures, &gcv);
    foreground = allocColor(xwin->getResource("Lcd", "Foreground"));
    background = allocColor(xwin->getResource("Lcd", "Background"));
    letterForeground =
      allocColor(xwin->getResource("Lcd.Letter", "Foreground"));
    letterForeground2 =
      allocColor(xwin->getResource("Lcd.Letter", "Foreground2"));	// ???
    letterBackground =
      allocColor(xwin->getResource("Lcd.Letter", "Background"));

    XWindowAttributes attr;
    XGetWindowAttributes(d, w, &attr);
    int i;
    for(i = 0; i < 96; i++)
      chars[i] = XCreatePixmap(d, w, 15, 21, attr.depth);
    for(i = 0x20; i < 0x80; i++) {
      XSetForeground(d, gc, letterBackground);
      XSetBackground(d, gc, letterBackground);
      XFillRectangle(d, chars[i-0x20], gc, 0, 0, 15, 21);
      XSetForeground(d, gc, letterForeground);
      int x, y;
      for(y = 0; y < 7; y++) {
	for(x = 0; x < 5; x++) {
	  if(lcdchars[(i-0x20)*7+y] & (1<<x)) {
	    XSetForeground(d, gc, letterForeground2);// ???
	    XFillRectangle(d, chars[i-0x20], gc, x*3, y*3, 3, 3);//???
	    XSetForeground(d, gc, letterForeground);
	    XFillRectangle(d, chars[i-0x20], gc, x*3, y*3, 2, 2);//???
//	    XFillRectangle(d, chars[i-0x20], gc, x*3, y*3, 3, 3);
	  }
	}
      }
    }

    int j, p;
    for(i = 0; i < 16; i++)
      for(j = 0; j < 16; j++)
	for(p = 0; p < 1+10 + 1; p++)
	  lcd_dd[p][i][j] = 0;
    memcpy(patch_name, "- SOUND Canvas -", 16);

    // Initial value of letter, reg->ldd is "- SOUND Canvas -",
    // or "              " ???
    // See patch common parameter to check.

    memcpy(letter, patch_name, 32);
    for(i = 16; i < 32; i++)
      letter[i] = ' ';
    memcpy(cur_letter, patch_name, 16);
    letter_length = display_length = 16;

    for(i = 0; i < MAX_PART; i++) {
      cur_vel[i] = 0;
      v[i] = v0[i] = 0;
    }

    XtVaSetValues(parent,
		  XtNx,  parent->core.width-LCDWidth,
		  NULL);
    XtAddEventHandler(lcd, ExposureMask, False,
		      (XtEventHandler)LCDDisplayExposeCB, (XtPointer)this);
    redraw();
  }

  ~LCDDisplay() {
    XFreeGC(d, gc);
    XtDestroyWidget(lcd);
  }

  operator Widget() { return lcd; }

  /*
   * allocate named color
   */
  unsigned long allocColor(const char* color) const {
    Colormap cmap;
    XColor c0, c1;
    cmap = DefaultColormap(d, 0);
    XAllocNamedColor(d, cmap, color, &c1, &c0);
    return(c1.pixel);
  }

  /*
   * just redraw
   */
  void redraw(void) const {
    int i, j, p;
    p = (limit_dot == 0) ? 0x0b : page;
    for(j = 0; j < 16; j++)
      for(i = 0; i < 16; i++)
	drawDot_(i, j, lcd_dd[p][i][j]);
    for(i = 0; i < 16; i++)
      drawALetter_(i, cur_letter[i]);
  }

  /*
   * draw displayed dot data
   */
  void draw(const unsigned char* data, unsigned char page_, unsigned time_) {
    if(data != NULL) {
      if(page != page_) {
	page = page_;
	limit_dot = gettime(480*time);
      }
      time = time_;
      int i, j;
      for(i = 0; i < 3; i++) {
	for(j = 0; j < 16; j++) {
	  drawDot(i*5+0, j, data[i*16+j] & 0x10);
	  drawDot(i*5+1, j, data[i*16+j] & 0x08);
	  drawDot(i*5+2, j, data[i*16+j] & 0x04);
	  drawDot(i*5+3, j, data[i*16+j] & 0x02);
	  drawDot(i*5+4, j, data[i*16+j] & 0x01);
	}
      }
      for(j = 0; j < 16; j++)
	drawDot(15, j, data[3*16+j] & 0x10);
    }
  }

  /*
   * draw displayed letter
   */
  void drawLetter(const char* data) {
    if(data != NULL) {
      int i;
      for(i = 0; (i < 32 && 0x20 <= (unsigned char)data[i] &&
		  (unsigned char)data[i] <= 0x7f); i++)
	letter[i] = data[i];
      for(/**/; i < 32; i++)
	letter[i] = ' ';
      for(i = 31; i >= 0 && letter[i] == ' '; i--);
      letter_length = i+1;

      if(letter_length <= 16) {
	for(i = 0; i < 16; i++)
	  drawALetter(i, letter[i]);
	limit_letter = gettime(300*8);	// 2.4sec
	display_length = 16;
      } else {
	// start scrolling
	limit_letter = gettime(300);	// 300ms
	display_length = 0;
	drawALetterWithScroll('<');
	flag = true;
      }
    }
  }

  /*
   * set patch name
   */
  void setPatchName(const char* data) {
    memcpy(patch_name, data, 16);
    // what will happen when patch name is set while redisplaying?
    if(limit_letter == 0) {
      memcpy(cur_letter, patch_name, 16);
      letter_length = display_length = 16;
      redraw();
    }
  }

  /*
   * update displayed letter
   */
  void update(void) {
    static char patch1[16];
    static char letter1[32];
    if(strncmp(patch1, (char*)reg->patch, 16)) {
      strcpy(patch1, (char*)reg->patch);
      setPatchName((char*)patch1);
    }
    if(strncmp(letter1, (char*)reg->lcd, 32)) {
      strncpy(letter1, (char*)reg->lcd, 32);
      drawLetter((char*)letter1);
    }
    draw((unsigned char*)&reg->lcd_dd[0], reg->page, reg->lcd_time);

    if(limit_letter > 0) {
      if(gettime(0) > limit_letter) {
	if(letter_length <= 16 && display_length == 16) {
	  int i;
	  for(i = 0; i < 16; i++)	// redisplaying patch name
	    drawALetter(i, patch_name[i]);
	  letter_length = display_length = 16;
	  limit_letter = 0;
	} else {
	  if(display_length < letter_length)
	    drawALetterWithScroll(letter[display_length]);
	  limit_letter = gettime(300);    // 300ms
	  if(++display_length >= letter_length) {
	    if(flag) {
	      if(display_length > letter_length) {
		// start displaying patch name
		memcpy(letter, patch_name, 32);
		letter_length = 16;
		display_length = 0;
		drawALetterWithScroll('<');
		flag = false;
	      }
	    } else {	// stop scrolling
	      letter_length = display_length = 16;
	      limit_letter = 0;
	    }
	  }
	}
      }
    }
    if(limit_dot > 0) {
      if(gettime(0) > limit_dot) {
	limit_dot = 0;
	// clear the level meter
	unsigned char page0 = page;
	page = 0x0b;
	int i, j;
	for(i = 0; i < 16; i++)
	  for(j = 0; j < 16; j++) {
	    lcd_dd[page][i][j] = 0;
	    drawDot_(i, j, 0);
	  }
	page = page0;
	for(i = 0; i < MAX_PART; i++) {
	  cur_vel[i] = 0;
	  v[i] = v0[i] = 0;
	}
      }
    } else {
      unsigned char page0 = page;
      page = 0x0b;
      update_velocity();
      page = page0;
    }
  }

private:
  /*
   * draw a dot
   */
  void drawDot(int i, int j, unsigned char data) {
    if(page <= 0x0a + 1) {
      if(lcd_dd[page][i][j] != data) {
	drawDot_(i, j, data);
	lcd_dd[page][i][j] = data;
      }
    }
  }

  /*
   * draw a dot actually
   */
  void drawDot_(int i, int j, unsigned char data) const {
    XSetForeground(d, gc, data ? foreground : background);
    XFillRectangle(d, w, gc,
		   i*LCDDotWidth, LCDLetterHeight+1+j*LCDDotHeight,
		   LCDDotWidth-1, LCDDotHeight-1);
  }

  /*
   * draw a letter with scroll
   */
  void drawALetterWithScroll(char data) {
    int i;
    for(i = 0; i < 15; i++)
      drawALetter(i, cur_letter[i+1]);
    drawALetter(15, data);
  }

  /*
   * draw a letter
   */
  void drawALetter(int i, char data) {
    if(cur_letter[i] != data) {
      drawALetter_(i, data);
      cur_letter[i] = data;
    }
  }

  /*
   * draw a letter actually
   */
  void drawALetter_(int i, char data) const {
    if(0x20 <= (unsigned char)data && (unsigned char)data <= 0x7f)  // ASCII
      XCopyArea(d, chars[data-0x20], w, gc,
		0, 0, LCDDotWidth-1, LCDLetterHeight, i*LCDDotWidth, 0);
  }

  /*
   * get time
   */
  long gettime(long msec) const {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return (tv.tv_sec & 0xffff)*1000 + tv.tv_usec/1000 + msec;
  }

  /*
   * pseudo level meter
   */
  void update_velocity(void) {
    int i;
    for(i = 0; i < nparts/*MAX_PART*/; i++) {
      draw_velocity(i, reg->vel[i]);
      if(cur_vel[i] > 0) {
	if(cur_vel[i] > VEL_FADE)
	  draw_velocity_(i, cur_vel[i]-VEL_FADE);
	else
	  cur_vel[i] = 0;
      }
    }
  }

  void draw_velocity(int part, unsigned char vel) {
    v0[part] = v[part];
    v[part] = vel;
    if(v0[part] != v[part])
      draw_velocity_(part, v[part]);
  }

  void draw_velocity_(int part, unsigned char vel) {
    if(vel == 0)	// note off
      return;
    unsigned char vel0 = cur_vel[part]/8;
    cur_vel[part] = vel;
    vel /= 8;
    if(vel0 == vel)
      return;
    int i;
    if(nparts <= 16) {
      if(vel0 < vel) {
	for(i = vel0; i < vel; i++)
	  drawDot(part, 15-i, 1);
      } else {
	for(i = vel; i < vel0; i++)
	  drawDot(part, 15-i, 0);
      }
    } else {
      if(nparts <= 32) {
	for(i = 0; i < 16; i++)
	  drawDot(i, 8-1, 1);	// partition ???
	if(part < 16) {
	  if(vel0 < vel) {
	    for(i = vel0/2; i < vel/2; i++)
	      drawDot(part, 8+7-i, 1);
	  } else {
	    for(i = vel/2; i < vel0/2; i++)
	      drawDot(part, 8+7-i, 0);
	  }
	} else {
	  if(vel0 < vel) {
	    for(i = vel0/2; i < vel/2; i++)
	      if(i < 7)
		drawDot(part-16, 8-i-2, 1);
	  } else {
	    for(i = vel/2; i < vel0/2; i++)
	      if(i < 7)
		drawDot(part-16, 8-i-2, 0);
	  }
	}
      } else {			// when nparts > 32 ???
      }
    }
  }
};


/*
 * Generic Label
 */
class Label : public BaseWidget {
private:
  unsigned long foreground, background;
  unsigned long illuminatedEdgeColor, shadowedEdgeColor;
  char* string;
  XFontStruct* fs;	// if NULL, fontset used
  XFontSet fontset;
  int fheight;		// the maximum height of the font set
  bool relief;		// whether a relief is made
  bool status;		// whether the label is shown
  int len;
  enum {Label_beggining, Label_center, Label_end} alignment;

public:
  Label(XWin* xwin_, Widget parent_, const char* name, int x_, int y_) :
  BaseWidget(xwin_, parent_, x_, y_), fheight(0), status(true) {
    const char* width_str = xwin->getResource(name, "Width");
    const char* height_str = xwin->getResource(name, "Height");
    width = atoi(width_str);
    height = atoi(height_str);
    if(!strcmp(xwin->getResource(name, "FontSet"), ""))
      fs = XLoadQueryFont(d, xwin->getResource(name, "Font"));
    else {
      fs = NULL;
      char** missing_charset_list;
      int missing_charset_count;
      char* def_string;
      fontset = XCreateFontSet(d, xwin->getResource(name, "FontSet"),
			       &missing_charset_list, &missing_charset_count,
			       &def_string);
      const XFontSetExtents* ext = XExtentsOfFontSet(fontset);
      fheight = ext->max_ink_extent.height;
    }
    const char* string_ = xwin->getResource(name, "labelString");
    len = strlen(string_);
    string = new char[len+1];
    strcpy(string, string_);
    foreground = allocColor(xwin->getResource(name, "Foreground"));
    background = allocColor(xwin->getResource(name, "Background"));
    illuminatedEdgeColor =
      allocColor(xwin->getResource(name, "illuminatedEdgeColor"));
    shadowedEdgeColor =
      allocColor(xwin->getResource(name, "shadowedEdgeColor"));
    relief = !strcmp(xwin->getResource(name, "Relief"), "True");
    const char* alignment_str = xwin->getResource(name, "Alignment");
    alignment = ((!strcmp(alignment_str, "") ||
		  !strcmp(alignment_str, "ALIGNMENT_BEGGINING"))
		 ? Label_beggining
		 : ((!strcmp(alignment_str, "ALIGNMENT_CENTER"))
		    ? Label_center
		    : ((!strcmp(alignment_str, "ALIGNMENT_END"))
		       ? Label_end
		       : Label_beggining)));	// default
    redraw();
  }

  ~Label(void) {
    hide();
    delete[] string;
    if(fs != NULL)
      XFreeFont(d, fs);
    else
      XFreeFontSet(d, fontset);
  }

  void show(void) {
    if(!status) {
      status = true;
      redraw();
    }
  }

  void hide(void) {
    if(status) {
      XSetForeground(d, gc, background);
      XFillRectangle(d, w, gc, x, y, width, height);
      status = false;
    }
  }

  void draw(const char* string_) {
    if(strcmp(string, string_)) {
      int newlen = strlen(string_);
      if(len != newlen) {
	delete[] string;
	string = new char[newlen+1];
	len = newlen;
      }
      strcpy(string, string_);
      redraw();
    }
  }

  void redraw(void) {
    if(status) {
      XSetForeground(d, gc, background);
      XFillRectangle(d, w, gc, x, y, width/*+1*/, height);	// +1???
      XSetForeground(d, gc, foreground);
      if(fs == NULL) {
	XmbDrawString(d, w, fontset, gc, x+(relief ? 2 : 0), y+fheight/*+2*/,
		      string, len);
      } else {
	XSetFont(d, gc, fs->fid);
	int textWidth;
	switch(alignment) {
	case Label_beggining:
	  XDrawString(d, w, gc, x+(relief ? 2 : 0), y+fs->ascent, string, len);
	  break;
	case Label_center:
	  textWidth = XTextWidth(fs, string, len);
	  XDrawString(d, w, gc, x+(width-textWidth-(relief ? 2 : 0)*2)/2,
		      y+fs->ascent, string, len);
	  break;
	case Label_end:
	  break;
	}
      }

      if(relief) {
	XSetForeground(d, gc, shadowedEdgeColor);
	XDrawLine(d, w, gc, x, y, x+width-1, y);
	XDrawLine(d, w, gc, x, y, x, y+height-1);
	XDrawLine(d, w, gc, x+2, y+height-2, x+width-2, y+height-2);
	XDrawLine(d, w, gc, x+width-2, y+2, x+width-2, y+height-2);
	XSetForeground(d, gc, illuminatedEdgeColor);
	XDrawLine(d, w, gc, x+1, y+height-1, x+width-1, y+height-1);
	XDrawLine(d, w, gc, x+width-1, y+1, x+width-1, y+height-1);
	XDrawLine(d, w, gc, x+1, y+1, x+width-2, y+1);
	XDrawLine(d, w, gc, x+1, y+1, x+1, y+height-2);
      }
    }
  }

  void setForeground(unsigned long foreground_) {
    foreground = foreground_;
  }
};


/*
 * Program Number
 */
class ProgramLabel : public Label {
private:
  volatile Register* reg;
  int part;
  int prg;

public:
  ProgramLabel(XWin* xwin_, Widget parent_, int x, int y,
	       volatile Register* reg_, int part_) :
  Label(xwin_, parent_, "Prg", x, y), reg(reg_), part(part_),
  prg(reg->prg[part]) {
    draw(prg);
  }

  void draw(int prg1) {
    prg = prg1;
    char no_str[4];
    no_str[0] = (prg+1>99) ? '1' : ' ';
    no_str[1] = (prg+1<10) ? ' ' : '0' + ((prg+1)/10)%10;
    no_str[2] = '0' + (prg+1)%10;
    no_str[3] = '\0';
    Label::draw(no_str);
  }

  void update(void) {
    int prg1 = reg->prg[part];
    if(prg1 != prg)
      draw(prg1);
  }

  void redraw(void) {
    Label::redraw();
  }
};


/*
 * GS Variation Number (GS)
 */
class BankLabel : public Label {
private:
  volatile Register* reg;
  int part;
  int bnk;

public:
  BankLabel(XWin* xwin_, Widget parent_, int x, int y,
	    volatile Register* reg_, int part_) :
  Label(xwin_, parent_, "Bnk", x, y), reg(reg_), part(part_),
  bnk(reg->bnk[part]) {
    draw(bnk);
  }

  void draw(int bnk1) {
    bnk = bnk1;
    char no_str[4];
    no_str[0] = (bnk>99) ? '1' : ' ';
    no_str[1] = (bnk<10) ? ' ' : '0' + (bnk/10)%10;
    no_str[2] = '0' + (bnk)%10;
    no_str[3] = '\0';
    Label::draw(no_str);
  }

  void update(void) {
    int bnk1 = reg->bnk[part];
    if(bnk1 != bnk)
      draw(bnk1);
  }

  void redraw(void) {
    Label::redraw();
  }
};


/*
 * Tone Map Number (SC-88, SC-88Pro)
 */
class MapLabel : public Label {
private:
  volatile Register* reg;
  int part;
  int map;

public:
  MapLabel(XWin* xwin_, Widget parent_, int x, int y,
	   volatile Register* reg_, int part_) :
  Label(xwin_, parent_, "Map", x, y), reg(reg_), part(part_),
  map(reg->map[part]) {
    draw(map);
  }

  void draw(int map1) {
    map = map1;
    char no_str[4];
    no_str[0] = (map>99) ? '1' : ' ';
    no_str[1] = (map<10) ? ' ' : '0' + (map/10)%10;
    no_str[2] = '0' + (map)%10;
    no_str[3] = '\0';
    Label::draw(no_str);
  }

  void update(void) {
    int map1 = reg->map[part];
    if(map1 != map)
      draw(map1);
  }

  void redraw(void) {
    Label::redraw();
  }
};


/*
 * Instrument or Drum Set Name
 */
class InstrumentLabel : public Label {
private:
  volatile Register* reg;
  int part;
  unsigned char prg;
  unsigned char bnk;
  unsigned char map;
  unsigned char rhythm;
  unsigned long capitalForeground, variationForeground, drumForeground;

public:
  InstrumentLabel(XWin* xwin_, Widget parent_, int x, int y,
		  volatile Register* reg_, int part_) :
  Label(xwin_, parent_, "Name", x, y), reg(reg_), part(part_),
  prg(reg->prg[part]), bnk(reg->bnk[part]), map(reg->map[part]),
  rhythm(reg->rhythm[part]) {
    capitalForeground =
      allocColor(xwin->getResource("Name", "capitalForeground"));
    variationForeground =
      allocColor(xwin->getResource("Name", "variationForeground"));
    drumForeground =
      allocColor(xwin->getResource("Name", "drumForeground"));
    draw(prg, bnk, map, rhythm);
  }

  void draw(unsigned char prg1, unsigned char bnk1, unsigned char map1,
	    unsigned char rhythm1) {
    prg = prg1;
    bnk = bnk1;
    map = map1;
    rhythm = rhythm1;
    const char* name;
    unsigned long color;
    if(rhythm != 0) {
      name = "STANDARD";
      if(reg->sc55 || reg->sc88 || reg->sc88pro) {
	name = ((map == 3) ? drum_sc88_pro[prg] :
		(map == 2) ? drum_sc88_88[prg] :
		(map == 1) ? drum_sc88_55[prg] :
		(map == 0) ? ((reg->sc88pro) ? drum_sc88_pro[prg] :
			      (reg->sc88) ? drum_sc88_88[prg] :
			      drum_sc88_55[prg]) :		// ???
		(char*)NULL);
	if((prg == 64 || prg == 65) && reg->userdrum[prg-64][0] != '\0')
	  name = (char*)reg->userdrum[prg-64];	// user drum set
      }
      if(reg->xg) {
	name = ((bnk ==   0) ? inst_xg[prg][map] :
		(bnk ==  48) ? inst_mu100[prg][map] :
		(bnk ==  64) ? sfx_xg[prg] :
		(bnk == 126) ? drum_sfx_xg[prg] :
		(bnk == 127) ? drum_xg[prg] :
		(char*)NULL);
      }
      if(name == NULL) {
	name = "No DRUM SET";
	color = drumForeground;
      } else {
	color = ((reg->xg && (bnk == 0 || bnk == 48 || bnk == 64)) ?
		 ((map == 0) ? capitalForeground : variationForeground) :
		 drumForeground);
      }
    } else {
      name = "Piano 1";
      if(reg->sc55 || reg->sc88 || reg->sc88pro) {
	name = ((map == 3) ? inst_sc88_pro[prg][bnk] :
		(map == 2) ? inst_sc88_88[prg][bnk] :
		(map == 1) ? inst_sc88_55[prg][bnk] :
		(map == 0) ? ((reg->sc88pro) ? inst_sc88_pro[prg][bnk] :
			      (reg->sc88) ? inst_sc88_88[prg][bnk] :
			      inst_sc88_55[prg][bnk]) :	// ???
		(char*)NULL);
      }
      if(reg->xg) {
	name = ((bnk ==   0) ? inst_xg[prg][map] :
		(bnk ==  48) ? inst_mu100[prg][map] :
		(bnk ==  64) ? sfx_xg[prg] :
		(bnk == 126) ? drum_sfx_xg[prg] :
		(bnk == 127) ? drum_xg[prg] :
		(char*)NULL);
      }
      if(name == NULL) {
	name = "NoInstrument";
	color = variationForeground;
      } else {
	color = (reg->xg ?
		 ((bnk == 126 || bnk == 127) ?
		  drumForeground :
		  (map == 0) ? capitalForeground : variationForeground) :
		 ((bnk == 0) ? capitalForeground : variationForeground));
      }
    }
    setForeground(color);
    Label::draw(name);
  }

  void update(void) {
    unsigned char prg1 = reg->prg[part];
    unsigned char bnk1 = reg->bnk[part];
    unsigned char map1 = reg->map[part];
    unsigned char rhythm1 = reg->rhythm[part];
    // Actually program doesn't change when only bank changed. But when
    // program changes to the same program number, we cannot detect it.
    // Therefore program changes even when only bank changed.
    if(prg1 != prg || /**/bnk1 != bnk ||/**/ map1 != map || rhythm1 != rhythm)
      draw(prg1, bnk1, map1, rhythm1);
    /*else
      if(bnk1 != bnk)
	bnk = bnk1;*/
  }

  void redraw(void) {
    Label::redraw();
  }
};


/*
 * Title
 */
class TitleLabel {
private:
  volatile Register* reg;
  Widget parent, form, l_title, title;
  const int max_title;
  char *string, *string1;

public:
  TitleLabel(Widget parent_, volatile Register* reg_) : reg(reg_),
  parent(parent_), max_title(64), string(new char[max_title+1]),
  string1(new char[max_title+1]) {
    form = XtVaCreateManagedWidget("titleForm", formWidgetClass, parent,
				   NULL);
    l_title = XtVaCreateManagedWidget("l_title", labelWidgetClass, form,
				      NULL);
    title = XtVaCreateManagedWidget("title", labelWidgetClass, form,
				    NULL);
    bzero(string, max_title+1);
    bzero(string1, max_title+1);
    XtVaSetValues(title, XtNlabel, " ", NULL);
    draw((const char*)reg->title);
  }

  ~TitleLabel() {
    XtDestroyWidget(title);
    XtDestroyWidget(l_title);
    XtDestroyWidget(form);
    delete[] string;
    delete[] string1;
  }

  void draw(const char* string_) {
    strncpy(string1, string_, max_title);
    if(strcmp(string, string1)) {
      if(string1[0] != '\0')
	strncpy(string, string1, max_title);
      else
	strncpy(string, " ", max_title);
      XtVaSetValues(title, XtNlabel, string, NULL);
    }
  }

  void update(void) {
    draw((const char*)reg->title);
  }
};


/*
 * Text
 */
class TextLabel : public Label {
private:
  volatile Register* reg;

public:
  TextLabel(XWin* xwin_, Widget parent_, int x, int y,
	    volatile Register* reg_) :
  Label(xwin_, parent_, "Text", x, y), reg(reg_) {
    draw((const char*)reg->text);
  }

  void update(void) {
    draw((const char*)reg->text);
  }

  void redraw(void) {
    Label::redraw();
  }
};


/*
 * EFX Type
 */
class EfxtypeLabel {
private:
  volatile Register* reg;
  Widget parent, form, l_efxtype, efxtype;
  unsigned char sc88pro;
  unsigned char type[2];

public:
  EfxtypeLabel(Widget parent_, volatile Register* reg_) :
  reg(reg_), parent(parent_), sc88pro(reg->sc88pro) {
    form = XtVaCreateManagedWidget("efxtypeForm", formWidgetClass, parent,
				   NULL);
    l_efxtype = XtVaCreateManagedWidget("l_efxtype", labelWidgetClass, form,
					NULL);
    efxtype = XtVaCreateManagedWidget("efxtype", labelWidgetClass, form,
				    NULL);
    type[0] = reg->efx_type[0];
    type[1] = reg->efx_type[1];
    draw(sc88pro, type[0], type[1]);
  }

  ~EfxtypeLabel() {
    XtDestroyWidget(efxtype);
    XtDestroyWidget(l_efxtype);
    XtDestroyWidget(form);
  }

  void draw(unsigned char sc88pro1, unsigned char type0, unsigned char type1) {
    sc88pro = sc88pro1;
    type[0] = type0;
    type[1] = type1;
    if(sc88pro) {
      const char* name = efx_sc88_pro[type[0]][type[1]];
      if(name == NULL)
	name = "No EFX";
      XtVaSetValues(efxtype, XtNlabel, name, NULL);
      XtSetMappedWhenManaged(efxtype, True);
    } else
      XtSetMappedWhenManaged(efxtype, False);
  }

  void update(void) {
    int sc88pro1 = reg->sc88pro;
    int type0 = reg->efx_type[0];
    int type1 = reg->efx_type[1];
    if(sc88pro != sc88pro1 || type[0] != type0 || type[1] != type1) {
      draw(sc88pro1, type0, type1);
    }
  }
};


/*
 * SC88 Label
 */
class SC88Label : public Label {
private:
  volatile Register* reg;
  unsigned char sc88;
  unsigned char sc88pro;

public:
  SC88Label(XWin* xwin_, Widget parent_, int x, int y,
	    volatile Register* reg_) :
  Label(xwin_, parent_, "l_sc88", x, y), reg(reg_), sc88(reg->sc88),
  sc88pro(reg->sc88pro) {
    draw(sc88, sc88pro);
  }

  void draw(unsigned char sc88_, unsigned char sc88pro_) {
    sc88 = sc88_;
    sc88pro = sc88pro_;
    if(sc88pro) {
      Label::draw("8P");
      Label::show();
    } else
      if(sc88) {
	Label::draw("88");
	Label::show();
      } else
	Label::hide();
  }

  void update(void) {
    int sc88_ = reg->sc88;
    int sc88pro_ = reg->sc88pro;
    if(sc88_ != sc88 || sc88pro_ != sc88pro)
      draw(sc88_, sc88pro_);
  }

  void redraw(void) {
    Label::redraw();
  }
};


/*
 * Sequencer Status
 */
class StatusLabel {
private:
  volatile Register* reg;
  Widget parent, form, l_status, statusW;
  unsigned char status;

public:
  StatusLabel(Widget parent_, volatile Register* reg_) :
  reg(reg_), parent(parent_), status(reg->status) {
    form = XtVaCreateManagedWidget("statusForm", formWidgetClass, parent,
				   NULL);
    l_status = XtVaCreateManagedWidget("l_status", labelWidgetClass, form,
				       NULL);
    statusW = XtVaCreateManagedWidget("status", labelWidgetClass, form,
				      NULL);
    draw(status);
  }

  ~StatusLabel() {
    XtDestroyWidget(statusW);
    XtDestroyWidget(l_status);
    XtDestroyWidget(form);
  }

  void draw(unsigned char status1) {
    const char* status_str[] = 
      {"IDLE", "PLAY", "PAUSE", "FADE OUT", "SEARCH", "COMPUTE"};
    status = status1;
    XtVaSetValues(statusW, XtNlabel, status_str[status], NULL);
  }

  void update(void) {
    unsigned char status1 = reg->status;
    if(status1 != status)
      draw(status1);
  }
};


/*
 * Timer
 */
class Timer {
private:
  volatile Register* reg;
  Widget parent, label;
  unsigned time;
  unsigned playtime;
  unsigned bar;

public:
  Timer(Widget parent_, volatile Register* reg_) :
  reg(reg_), parent(parent_), time(reg->time), playtime(reg->playtime),
  bar(reg->bar) {
    label = XtVaCreateManagedWidget("timer", labelWidgetClass, parent,
				    XtNtop,         XawChainBottom,
				    XtNbottom,      XawChainBottom,
				    XtNleft,        XawChainLeft,
				    XtNright,       XawChainLeft,
				    XtNborderWidth, 0,
				    NULL);
    draw(time, playtime, bar);
  }

  ~Timer() {
    XtDestroyWidget(label);
  }

  void draw(unsigned time1, unsigned playtime1, unsigned bar1) {
    time = time1;
    playtime = playtime1;
    bar = bar1;
    char time_str[256];
    sprintf(time_str, "Time: %02d:%02d:%02d / %02d:%02d:%02d  Bar: %04d",
	    time/6000%60,     time/100%60,     time%100,
	    playtime/6000%60, playtime/100%60, playtime%100,
	    bar);
    XtVaSetValues(label, XtNlabel, time_str, NULL);
  }

  void update(void) {
    unsigned time1 = reg->time;
    unsigned playtime1 = reg->playtime;
    unsigned bar1 = reg->bar;
    if(time1 != time || playtime1 != playtime || bar1 != bar);
      draw(time1, playtime1, bar1);
  }
};


/*
 * Logo
 */
class Logo {
private:
  volatile unsigned char* reg;
  Widget parent, label;
  unsigned char v;

public:
  Logo(XWin* xwin, const char* name, Widget parent_,
       volatile unsigned char* reg_)
    : reg(reg_), parent(parent_), v(*reg) {
    Pixmap pixmap;
    int width, height;
    xwin->readXpm(xwin->getResource(name), &pixmap, &width, &height);
    label =
      XtVaCreateManagedWidget(name, labelWidgetClass, parent,
			      XtNwidth,  width,
			      XtNheight, height,
			      XtNbitmap, pixmap,
			      XtNmappedWhenManaged, False,
			      NULL);
//    draw(v);
  }

  void draw(unsigned char v1) {
    v = v1;
    if(v)
      XtMapWidget(label);
    else
      XtUnmapWidget(label);
  }

  void update(void) {
    unsigned char v1 = *reg;
    if(v1 != v)
      draw(v1);
  }
};


//enum Part_ { PartLabel, TopPart, MiddlePart, BottomPart, MasterPart };

/*
 * Part
 */
class Part : public BaseWidget {
private:
  volatile Register* reg;
  int no;
  int n_widget;
  BaseWidget* widget[15];
  const char* order;

public:
  Part(XWin* xwin_, Widget parent_, int x_, int y_,
       volatile Register* reg_, int no_) :
  BaseWidget(xwin_, parent_, x_, y_), reg(reg_), no(no_), n_widget(0) {
    width = 0;
    height = 0;

    int w, h;
    char no_str[3];
    no_str[0] = (no+1>9) ? '0'+(no+1)/10 : ' ';
    no_str[1] = '0' + (no+1)%10;
    no_str[2] = '\0';
    widget[n_widget] = new Label(xwin, parent, "Ch", x+width, y);
    ((Label*)widget[n_widget])->draw(no_str);
    widget[n_widget++]->getSize(w, h);
    width += w;
    if(h > height)
      height = h;

    // create widgets
    order = xwin->getResource("Part", "Order");
    const char* p = order;
    int i;
    int len = 0;
    for(i = 0; i < 16; i++) {
      if(i > 0) {
	if(p[len] == '\0')
	  break;
	else
	  p += len+1;
      }
      len = 0;
      while(p[len] != ',' && p[len] != '\0')
	len++;
      if(!strncmp("vel", p, len)) {
	widget[n_widget] = new Velocity(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("vol", p, len)) {
	widget[n_widget] = new VolumeMeter(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("expr", p, len)) {
	widget[n_widget] =
	  new ExpressionMeter(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("pan", p, len)) {
	widget[n_widget] = new PanpotWheel(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("pitch", p, len)) {
	widget[n_widget] = new BenderWheel(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("mod", p, len)) {
	widget[n_widget] =
	  new ModulationMeter(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("rev", p, len)) {
	widget[n_widget] = new ReverbMeter(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("cho", p, len)) {
	widget[n_widget] = new ChorusMeter(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("dly", p, len)) {
	widget[n_widget] = new DelayMeter(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("prg", p, len)) {
	widget[n_widget] = new ProgramLabel(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("bnk", p, len)) {
	widget[n_widget] = new BankLabel(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("map", p, len)) {
	widget[n_widget] = new MapLabel(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("name", p, len)) {
	widget[n_widget] =
	  new InstrumentLabel(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      if(!strncmp("keyboard", p, len)) {
	widget[n_widget] = new Keyboard(xwin, parent, x+width, y, reg, no);
	goto next;
      }
      // error
      {
	char* widget_str = new char[len+1];
	strncpy(widget_str, p, len);
	widget_str[len] = '\0';
	fprintf(stderr, "invalid widget name `%s\'\n", widget_str);
	delete[] widget_str;
	continue;
      }

    next:
      widget[n_widget++]->getSize(w, h);
      width += w;
      if(h > height)
	height = h;
    }
    if(n_widget == 1) {		// except ch
      fprintf(stderr, "There are no widgets available.\n");
      cleanup(0);
    }
  }

  ~Part() {
    int i;
    for(i = 0; i < n_widget; i++)
      delete widget[i];
  }

  void update(void) {
    int i;
    for(i = 0; i < n_widget; i++)
      widget[i]->update();
  }

  void redraw(void) {
    int i;
    for(i = 0; i < n_widget; i++)
      widget[i]->redraw();
  }
};


/*
 * Master Part
 */
class MasterPart : public BaseWidget {
private:
  volatile Register* reg;
  int n_widget;
  BaseWidget* widget[15];
  const char* order;

public:
  MasterPart(XWin* xwin_, Widget parent_, int x_, int y_,
	     volatile Register* reg_) :
  BaseWidget(xwin_, parent_, x_, y_), reg(reg_), n_widget(0) {
    width = 0;
    height = 0;

    int w, h;
    widget[n_widget] = new Label(xwin, parent, "l_master", x+width, y);
    widget[n_widget++]->getSize(w, h);
    width += w;
    if(h > height)
      height = h;

    // create widgets
    order = xwin->getResource("Part", "Order");
    const char* p = order;
    int i;
    int len = 0;
    for(i = 0; i < 16; i++) {
      if(i > 0) {
	if(p[len] == '\0')
	  break;
	else
	  p += len+1;
      }
      len = 0;
      while(p[len] != ',' && p[len] != '\0')
	len++;
      if(!strncmp("vol", p, len)) {
	widget[n_widget] = new MVolumeMeter(xwin, parent, x+width, y, reg);
	goto next;
      }
      if(!strncmp("pan", p, len)) {
	widget[n_widget] = new MPanpotWheel(xwin, parent, x+width, y, reg);
	goto next;
      }
      if(!strncmp("vel", p, len)) {
	width += VelocityMeterWidth;
	continue;
      }
      if(!strncmp("expr", p, len) ||
	 !strncmp("rev", p, len) ||
	 !strncmp("cho", p, len) ||
	 !strncmp("dly", p, len)) {
	width += MeterWidth;
	continue;
      }
      if(!strncmp("pitch", p, len) ||
	 !strncmp("mod", p, len)) {
	width += WheelWidth;
	continue;
      }
      if(!strncmp("prg", p, len)) {
	width += atoi(xwin->getResource("Prg", "Width"));
	continue;
      }
      if(!strncmp("bnk", p, len)) {
	width += atoi(xwin->getResource("Bnk", "Width"));
	continue;
      }
      if(!strncmp("map", p, len)) {
	width += atoi(xwin->getResource("Map", "Width"));
	continue;
      }
      if(!strncmp("name", p, len)) {
	width += atoi(xwin->getResource("Name", "Width"));
	continue;
      }
      if(!strncmp("keyboard", p, len)) {
	width += KeyboardWidth;
	continue;
      }
      // error
      continue;

    next:
      widget[n_widget++]->getSize(w, h);
      width += w;
      if(h > height)
	height = h;
    }
  }

  ~MasterPart() {
    int i;
    for(i = 0; i < n_widget; i++)
      delete widget[i];
  }

  void update(void) {
    int i;
    for(i = 0; i < n_widget; i++)
      widget[i]->update();
  }

  void redraw(void) {
    int i;
    for(i = 0; i < n_widget; i++)
      widget[i]->redraw();
  }
};


/*
 * Part Labels
 */
class PartLabels : public BaseWidget {
private:
  int n_widget;
  Label* widget[15];
  const char* order;

public:
  PartLabels(XWin* xwin_, Widget parent_, int x_, int y_) :
  BaseWidget(xwin_, parent_, x_, y_), n_widget(0) {
    width = 0;
    height = 0;

    int w, h;
    widget[n_widget] = new Label(xwin, parent, "l_ch", x+width, y);
    widget[n_widget++]->getSize(w, h);
    width += w;
    if(h > height)
      height = h;

    // create widgets
    order = xwin->getResource("Part", "Order");
    const char* p = order;
    int i;
    int len = 0;
    for(i = 0; i < 16; i++) {
      if(i > 0) {
	if(p[len] == '\0')
	  break;
	else
	  p += len+1;
      }
      len = 0;
      while(p[len] != ',' && p[len] != '\0')
	len++;
      if(!strncmp("vel", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_vel", x+width, y);
	goto next;
      }
      if(!strncmp("vol", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_vol", x+width, y);
	goto next;
      }
      if(!strncmp("expr", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_expr", x+width, y);
	goto next;
      }
      if(!strncmp("pan", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_pan", x+width, y);
	goto next;
      }
      if(!strncmp("pitch", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_pitch", x+width, y);
	goto next;
      }
      if(!strncmp("mod", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_mod", x+width, y);
	goto next;
      }
      if(!strncmp("rev", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_rev", x+width, y);
	goto next;
      }
      if(!strncmp("cho", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_cho", x+width, y);
	goto next;
      }
      if(!strncmp("dly", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_dly", x+width, y);
	goto next;
      }
      if(!strncmp("prg", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_prg", x+width, y);
	goto next;
      }
      if(!strncmp("bnk", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_bnk", x+width, y);
	goto next;
      }
      if(!strncmp("map", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_map", x+width, y);
	goto next;
      }
      if(!strncmp("name", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_name", x+width, y);
	goto next;
      }
      if(!strncmp("keyboard", p, len)) {
	widget[n_widget] = new Label(xwin, parent, "l_C1", x+width, y);
	widget[n_widget++]->getSize(w, h);
	width += KeyboardWidth;
	if(h > height)
	  height = h;
	continue;
      }
      // error
      continue;

    next:
      widget[n_widget++]->getSize(w, h);
      width += w;
      if(h > height)
	height = h;
    }
  }

  ~PartLabels() {
    int i;
    for(i = 0; i < n_widget; i++)
      delete widget[i];
  }

  void update(void) {
    int i;
    for(i = 0; i < n_widget; i++)
      widget[i]->update();
  }

  void redraw(void) {
    int i;
    for(i = 0; i < n_widget; i++)
      widget[i]->redraw();
  }
};


/*
 * Menu Bar
 */
class MenuBar {
private:
  Widget menubar;
  Widget menu[2];
  enum fileMenu {fileMenu_Open = 0, fileMenu_Quit = 1};
  enum instMenu {instMenu_sc55 = 0, instMenu_sc88 = 1,
		 instMenu_sc88pro = 2, instMenu_xg = 3};
  Widget fileButton[2];
  Widget instButton[4];
  static const unsigned char check_bits[8*16];
  Pixmap check;
  int radio;

public:
  MenuBar(Widget parent);
  ~MenuBar();
  operator Widget() { return menubar; }
  friend void FileCB(Widget w, XtPointer index, XtPointer call_data);
  friend void InstCB(Widget w, XtPointer index, XtPointer call_data);
};


const unsigned char MenuBar::check_bits[8*16] = {
  0x00, 0x00, 0x00, 0x80, 0x40, 0x40, 0x20, 0x20, 0x20, 0x20, 0x30, 0x13,
  0x16, 0x1c, 0x18, 0x18
};


MenuBar::MenuBar(Widget parent)
: radio(0)
{
#ifdef USE_MOTIF
  menubar = XmVaCreateSimpleMenuBar(parent, "menubar",
				    XmVaCASCADEBUTTON, NULL, '\0',
				    XmNbuttonCount, 1,
				    NULL);
  menu[0] =
    XmVaCreateSimplePulldownMenu(menubar,
				 "FileMenu", 0, FileCB,
				 XmVaPUSHBUTTON, NULL, '\0', NULL, NULL,
				 XmVaSEPARATOR,
				 XmVaPUSHBUTTON, NULL, '\0', NULL, NULL,
				 NULL);
  XtManageChild(menubar);
#else
  menubar = XtVaCreateManagedWidget("menubar", boxWidgetClass, parent,
				    XtNorientation, XtEhorizontal,
				    XtNtop,         XawChainTop,
				    XtNbottom,      XawChainTop,
				    XtNleft,        XawChainLeft,
				    XtNright,       XawChainLeft,
				    XtNborderWidth, 0,
				    NULL);
  // File
  menu[0] =
    XtVaCreateManagedWidget("FileMenu", menuButtonWidgetClass, menubar, NULL);
  Widget menushell0 =
    XtVaCreatePopupShell("menu", simpleMenuWidgetClass, menu[0], NULL);
  fileButton[0] =
    XtVaCreateManagedWidget("Open", smeBSBObjectClass, menushell0,
			    XtNsensitive, False,
			    NULL);
  XtVaCreateManagedWidget("line", smeLineObjectClass, menushell0, NULL);
  fileButton[1] =
    XtVaCreateManagedWidget("Quit", smeBSBObjectClass, menushell0, NULL);
  XtAddCallback(fileButton[0], XtNcallback, FileCB, (XtPointer)fileMenu_Open);
  XtAddCallback(fileButton[1], XtNcallback, FileCB, (XtPointer)fileMenu_Quit);
  // Instrument
  menu[1] =
    XtVaCreateManagedWidget("InstMenu", menuButtonWidgetClass, menubar,
			    XtNsensitive, False,
			    NULL);
  Widget menushell1 =
    XtVaCreatePopupShell("menu", simpleMenuWidgetClass, menu[1], NULL);
  instButton[0] =
    XtVaCreateManagedWidget("sc55", smeBSBObjectClass, menushell1,
                            XtNleftMargin, 8,
			    NULL);
  instButton[1] =
    XtVaCreateManagedWidget("sc88", smeBSBObjectClass, menushell1,
                            XtNleftMargin, 8,
			    NULL);
  instButton[2] =
    XtVaCreateManagedWidget("sc88pro", smeBSBObjectClass, menushell1,
                            XtNleftMargin, 8,
			    NULL);
  instButton[3] =
    XtVaCreateManagedWidget("xg", smeBSBObjectClass, menushell1,
                            XtNleftMargin, 8,
			    NULL);
  XtAddCallback(instButton[0], XtNcallback,
		InstCB, (XtPointer)instMenu_sc55);
  XtAddCallback(instButton[1], XtNcallback,
		InstCB, (XtPointer)instMenu_sc88);
  XtAddCallback(instButton[2], XtNcallback,
		InstCB, (XtPointer)instMenu_sc88pro);
  XtAddCallback(instButton[3], XtNcallback,
		InstCB, (XtPointer)instMenu_xg);
  check = XCreateBitmapFromData(XtDisplay(menubar),
				XRootWindowOfScreen(XtScreen(menubar)),
				(const char*)check_bits, 8, 16);
  XtVaSetValues(instButton[2], XtNleftBitmap, check, NULL);
#endif
}


MenuBar::~MenuBar()
{
}


/*
 * Body
 */
class Body {
private:
  Display* d;
  Window w;
  XWin* xwin;
  Widget toplevel;
  Widget parent;
  Widget body;
//  int n_part;	// number of parts displayed

  int nparts;
  Part** part;
  MasterPart* masterPart;
  PartLabels* partLabels;
  SC88Label* sc88Label;
  TextLabel* textLabel;
//Label* memo[12];
//Label* track[MIDI_CHANNEL];

  int width, height;
  GC gc;

public:
  Body(XWin* xwin_, Widget parent_, Widget toplevel_);
  ~Body();
  void initialize(void);
  void main(void);
  void update(void);
  void redraw(void);

  operator Widget() { return body; }

  /*
   * allocate named color
   */
  unsigned long allocColor(const char* color) const {
    Colormap cmap;
    XColor c0, c1;
    cmap = DefaultColormap(XtDisplay(parent), 0);
    XAllocNamedColor(XtDisplay(parent), cmap, color, &c1, &c0);
    return(c1.pixel);
  }
};


XWin::XWin(int argc, char* argv[])
: d(NULL), mb(NULL), title(NULL), body(NULL), status(NULL), efxtype(NULL),
  timer(NULL), gmLogo(NULL), gsLogo(NULL), xgLogo(NULL), lcd(NULL),
  wmname(NULL)
{
//  setlocale(LC_ALL, "");
  XtSetLanguageProc(NULL, NULL, NULL);

  toplevel = XtVaAppInitialize(&appcontext, CLASS_NAME, NULL, 0, &argc, argv,
			       fallback_resources, NULL);

  d = XtDisplay(toplevel);

  char* res = XResourceManagerString(d);
  if(res == NULL)
    rdb = NULL;
  else {
    rdb = XrmGetStringDatabase(res);
    XFree(res);
  }

  rdb2 = XrmGetDatabase(d);
  int i;
  for(i = 0; fallback_resources[i] != NULL; i++)
    XrmPutLineResource(&rdb2, fallback_resources[i]);

  const char* home_dir;
  char rdbfile[MAX_PATH+MAX_NAME+1];
  if((home_dir = getenv("HOME")) != NULL) {
    sprintf(rdbfile, "%s/.Xdefaults", home_dir);
    XrmCombineFileDatabase(rdbfile, &rdb2, True);
  }

  if(rdb != NULL)
    XrmCombineDatabase(rdb2, &rdb, False);
  else
    rdb = rdb2;
  XrmSetDatabase(d, rdb);

/*  XtVaGetApplicationResources(toplevel, (XtPointer)&globRes,
			      gres, XtNumber(gres), NULL);*/

  paned = XtVaCreateManagedWidget("paned", panedWidgetClass, toplevel,
				  XtNdefaultDistance, 0,
				  NULL);
  XtAppAddActions(appcontext, actions, XtNumber(actions));
  mb = new MenuBar(paned);
  title = new TitleLabel(paned, reg);		
  body = new Body(this, paned, toplevel);
  paned1 = XtVaCreateManagedWidget("paned1", panedWidgetClass, paned,
				   XtNdefaultDistance, 0,
				   XtNorientation,     XtorientHorizontal,
				   NULL);
  paned2 = XtVaCreateManagedWidget("paned2", panedWidgetClass, paned1,
				   XtNdefaultDistance, 0,
				   XtNorientation,     XtorientVertical,
				   NULL);
  status = new StatusLabel(paned2, reg);
  efxtype = new EfxtypeLabel(paned2, reg);
  timer = new Timer(paned2, reg);
  logos = XtVaCreateManagedWidget("logos", formWidgetClass, paned2,
				  XtNborderWidth, 0,
				  NULL);
  gmLogo = new Logo(this, "gmLogo", logos, &reg->gm);
  gsLogo = new Logo(this, "gsLogo", logos, &reg->gs);
  xgLogo = new Logo(this, "xgLogo", logos, &reg->xg);
  buttons = XtVaCreateManagedWidget("buttons", boxWidgetClass, paned2,
				    NULL);
  lcd = new LCDDisplay(this, paned1, MIDI_CHANNEL, reg);	// ???

//  w = XtWindow(paned);
  w = XRootWindowOfScreen(XtScreen(toplevel));

  playPixmap =
    XCreateBitmapFromData(d, w, (const char*)play_bits,
			  play_width, play_height);
  pausePixmap =
    XCreateBitmapFromData(d, w, (const char*)pause_bits,
			  pause_width, pause_height);
  stopPixmap =
    XCreateBitmapFromData(d, w, (const char*)stop_bits,
			  stop_width, stop_height);
  backPixmap =
    XCreateBitmapFromData(d, w, (const char*)slow_bits,
			  slow_width, slow_height);
  fwrdPixmap =
    XCreateBitmapFromData(d, w, (const char*)fwrd_bits,
			  fwrd_width, fwrd_height);
  prevPixmap =
    XCreateBitmapFromData(d, w, (const char*)prev_bits,
			  prev_width, prev_height);
  nextPixmap =
    XCreateBitmapFromData(d, w, (const char*)next_bits,
			  next_width, next_height);
  quitPixmap =
    XCreateBitmapFromData(d, w, (const char*)quit_bits,
			  quit_width, quit_height);
  playButton =
    XtVaCreateManagedWidget("play", commandWidgetClass, buttons,
			    XtNbitmap, playPixmap,
			    NULL);
  pauseButton =
    XtVaCreateManagedWidget("pause", toggleWidgetClass, buttons,
			    XtNbitmap, pausePixmap,
			    NULL);
  stopButton =
    XtVaCreateManagedWidget("stop", commandWidgetClass, buttons,
			    XtNbitmap, stopPixmap,
			    NULL);
  backButton =
    XtVaCreateManagedWidget("back", toggleWidgetClass, buttons,
			    XtNbitmap, backPixmap,
			    NULL);
  fwrdButton =
    XtVaCreateManagedWidget("fwrd", toggleWidgetClass, buttons,
			    XtNbitmap, fwrdPixmap,
			    NULL);
  prevButton =
    XtVaCreateManagedWidget("prev", commandWidgetClass, buttons,
			    XtNbitmap, prevPixmap,
			    NULL);
  nextButton =
    XtVaCreateManagedWidget("next", commandWidgetClass, buttons,
			    XtNbitmap, nextPixmap,
			    NULL);
  quitButton =
    XtVaCreateManagedWidget("quit", commandWidgetClass, buttons,
			    XtNbitmap, quitPixmap,
			    NULL);

  Dimension button_height;
  XtVaGetValues(buttons, XtNheight, &button_height, NULL);
//  XtVaSetValues(buttons, XtNmax, button_height, NULL);
//  XtVaSetValues(buttons, XtNmin, button_height, NULL);

  XtRealizeWidget(toplevel);

  body->initialize();
  lcd->initialize();
  gmLogo->draw(0);	// ???
  gsLogo->draw(0);
  xgLogo->draw(0);

//  XawPanedSetRefigureMode(paned, True);
//  XawPanedSetRefigureMode(paned1, True);
//  XawPanedSetRefigureMode(paned2, True);

//  XSizeHints sizehints;
//  sizehints.flags = 0;
//  XSetWMNormalHints(d, w, &sizehints);

//  XClassHint classhints = {RESOURCE_NAME, CLASS_NAME};
//  XSetClassHint(d, w, &classhints);

  XWMHints wmhints;
  wmhints.flags = IconPixmapHint;
  int icon_width, icon_height;
  const char* iconPixmap_str = getResource("iconPixmap");
  if(strcmp(iconPixmap_str, "") &&
     readXpm(iconPixmap_str, &wmhints.icon_pixmap,
	     &icon_width, &icon_height))
    XSetWMHints(d, XtWindow(toplevel), &wmhints);

  XtAddCallback(playButton,  XtNcallback, ControlCB, "play");
  XtAddCallback(pauseButton, XtNcallback, ControlCB, "pause");
  XtAddCallback(stopButton,  XtNcallback, ControlCB, "stop");
  XtAddCallback(backButton,  XtNcallback, ControlCB, "slow");	// ???
  XtAddCallback(fwrdButton,  XtNcallback, ControlCB, "fast");	// ???
  XtAddCallback(prevButton,  XtNcallback, ControlCB, "previous");
  XtAddCallback(nextButton,  XtNcallback, ControlCB, "next");
  XtAddCallback(quitButton,  XtNcallback, ControlCB, "clear");

  XtAppAddTimeOut(appcontext, SLEEP_TIME_PLAYING, &::timer, 0);
}


XWin::~XWin()
{
  delete lcd;
  delete gmLogo;
  delete gsLogo;
  delete xgLogo;
  delete efxtype;
  delete status;
  delete body;
  delete title;
  delete mb;
  XrmDestroyDatabase(rdb);
  XtDestroyApplicationContext(appcontext);
  exit(EXIT_SUCCESS);
}


/*
 *
 */
void XWin::main(void)
{
  XtAppMainLoop(appcontext);
}


Body::Body(XWin* xwin_, Widget parent_, Widget toplevel_)
: xwin(xwin_), toplevel(toplevel_), parent(parent_), 
  nparts(0), part(NULL), masterPart(NULL),
  partLabels(NULL), sc88Label(NULL), textLabel(NULL),
  width(0), height(0)
{
  body = XtVaCreateManagedWidget("body", coreWidgetClass, parent,
				 XtNwidth,       640,	// ???
				 XtNheight,      480,	// ???
				 XtNborderWidth, 0,
				 NULL);
  d = XtDisplay(body);
  w = XtWindow(body);
  XtAddEventHandler(body, ExposureMask, False,
		    (XtEventHandler)BodyExposeCB, (XtPointer)this);
}


Body::~Body()
{
  delete partLabels;
  int i;
  for(i = 0; i < nparts; i++)
    delete part[i];
  delete[] part;
  delete masterPart;
  delete textLabel;
  delete sc88Label;
//  delete l_track;
}


/*
 *
 */
void Body::initialize(void)
{
  XGCValues gcv;
  gcv.graphics_exposures = False;
  gc = XCreateGC(d, XRootWindowOfScreen(XtScreen(parent)),
		 GCGraphicsExposures, &gcv);

  if(reg->status == STATUS_PLAY) {
#if defined(WMNAME_FILENAME)
    xwin->changeTitle((char*)reg->file);
#elif defined(WMNAME_TITLE)
    xwin->changeTitle((char*)reg->title);
#endif
  } else
    xwin->changeTitle(version);

  int w_, h;
//  titleLabel = new TitleLabel(xwin, body, 0/*l_title->getRight()*/, 0, reg);
//  titleLabel->getSize(w_, h);
//  height += h;
  partLabels = new PartLabels(xwin, body, 0, height);
  partLabels->getSize(w_, h);
  height += h;

//l_track = new Label(this, "l_track", X_TRACK_NAME, Y_PARA_NAME);

  nparts = atoi(xwin->getResource("Nparts"));
  if(nparts <= 0 || nparts > MAX_PART) {
    fprintf(stderr, "invalid number of parts specified.\n");
    cleanup(0);
  }
  part = new Part*[nparts];
  int i;
  for(i = 0; i < nparts; i++) {
    part[i] = new Part(xwin, body, 0, height, reg, i);
    part[i]->getSize(w_, h);
    width = w_;
    height += h;
  }
  masterPart = new MasterPart(xwin, body, 0, height, reg);
  masterPart->getSize(w_, h);
  height += h;

//  for(i = 0; i < nparts; i++)
//    track[i] = new Label(this, "track", X_KEY, Y_CH(i));
//    for(i = 0; i < 12; i++)
//      memo[i] = new Label(this, "memo", X_KEY, Y_CH(i+2));

  textLabel =
    new TextLabel(xwin, body, width-12*20, masterPart->getTop(), reg);
//  efxLabel = new EfxtypeLabel(xwin, body, 0, masterPart->getBottom(), reg);
//  statusLabel = new StatusLabel(xwin, body, 0, efxLabel->getBottom(), reg);
  height = textLabel->getBottom();

//  sc88Label = new SC88Label(xwin, body, xgLabel->getRight()+1, 0, reg);

  XtVaSetValues(toplevel,
		XtNwidth,  toplevel->core.width+(width-640),
		XtNheight, toplevel->core.height+(height-480),
		NULL);
  XtVaSetValues(parent,
		XtNwidth,  parent->core.width+(width-640),
		XtNheight, parent->core.height+(height-480),
		NULL);
  XtVaSetValues(body,
		XtNwidth,  width,
		XtNheight, height,
		NULL);
  XtVaSetValues(body, XtNmax, height, NULL);
  XtVaSetValues(body, XtNmin, height, NULL);
}


/*
 * main loop
 */
void Body::main(void)
{
  static unsigned long sleep_time = SLEEP_TIME_IDLING;
  static int status, status0 = STATUS_IDLE;

  status = reg->status;
  update();
  if(status != status0) {
    switch(status) {
    case STATUS_IDLE:
      sleep_time = SLEEP_TIME_IDLING;
      xwin->changeTitle(NULL);
      break;
    case STATUS_PLAY:
      if(status0 == STATUS_IDLE || status0 == STATUS_COMP) {
	sleep_time = SLEEP_TIME_PLAYING;
#if defined(WMNAME_FILENAME)
	xwin->changeTitle((char*)reg->file);
#elif defined(WMNAME_TITLE)
	xwin->changeTitle((char*)reg->title);
#endif
      }
      break;
    }
  }
  XFlush(XtDisplay(parent));
  status0 = status;

  XtAppAddTimeOut(xwin->appcontext, sleep_time, &timer, 0);
}


/*
 *
 */
void Body::update(void)
{
  int i;
  for(i = 0; i < nparts; i++)
    part[i]->update();
  masterPart->update();
  textLabel->update();
//DrawTrackName(reg0.name[0]);
//  sc88Label->update();
}


/*
 *
 */
void Body::redraw(void)
{
  int i;
  for(i = 0; i < nparts; i++)
    part[i]->redraw();
  masterPart->redraw();
  textLabel->redraw();
//DrawTrackName(reg0.name[0]);
//  sc88Label->redraw();
  partLabels->redraw();
//l_track->redraw();
}


/*
 * print version number
 */
inline void print_version(void)
{
  fprintf(stderr,
	  "MIDI status display %s Copyright (C) 1995-1998 S.Kuramochi\n",
	  version);
}


/*
 * print usage
 */
inline void print_usage(void)
{
  fprintf(stderr,
	  "usage: %s [-options]\n"
	  " -h --help           show this help\n"
	  " -V --version        display version number\n"
/*	  " --sc55              use SC-55 MAP when TONE MAP NUMBER is 0\n"
	  " --sc88              use SC-88 MAP when TONE MAP NUMBER is 0 (default)\n"
	  " --sc88pro           use SC-88Pro MAP when TONE MAP NUMBER is 0\n"*/
	  " -display display    X server to use\n"
	  " -geometry geometry  initial size and location\n"
	  ,
	  prgname);
}


static XWin* xwin = NULL;

/*
 * main routine
 */
int main(int argc, char* argv[])
{
  prgname = argv[0];
  struct option long_options[] = {
    {"display",	 required_argument, NULL, 'D'},
    {"geometry", required_argument, NULL, 'G'},
    {"help",     no_argument,       NULL, 'h'},
    {"iconic",	 no_argument,       NULL, 'I'},
/*  {"sc55",     no_argument,       NULL, '5'},
    {"sc88",     no_argument,       NULL, '8'},
    {"sc88pro",  no_argument,       NULL, 'P'},*/
    {"version",  no_argument,       NULL, 'V'},
    {"xrm",      required_argument, NULL, 'R'},
    {NULL,       no_argument,       NULL,  0 }
  };
  optind = 0;
  opterr = 1;
  optopt = '?';
  int c;
  for(;;) {
    int option_index = 0;
    if((c = getopt_long_only(argc, argv, "hV",
			     long_options, &option_index)) == EOF)
      break;
    switch(c) {
#if 0
    case '5': // use SC-55 MAP when TONE MAP NUMBER is 0
      break;
    case '8': // use SC-88 MAP when TONE MAP NUMBER is 0
      break;
    case 'P': // use SC-88Pro MAP when TONE MAP NUMBER is 0
      break;
#endif
    case 'h': // show help
      print_version();
      print_usage();
      exit(EXIT_SUCCESS);
      break;
    case 'V': // display version number
      print_version();
      fprintf(stderr, 
	      "This is free software with ABSOLUTELY NO WARRANTY.\n"
	      "For details please see the file COPYING.\n");
      exit(EXIT_SUCCESS);
      break;
    case 'D': // X server to use
    case 'G': // geometry
    case 'I': // iconic
    case 'R': // resource
      break;
    case ':': // missing parameter
    case '?': // unknown option character
      print_usage();
      exit(EXIT_FAILURE);
      break;
    default:
      fprintf(stderr,
	      "getopt_long_only() returned character code %02x\n", c&0xff);
      break;
    }
  }
  if(optind <= argc)
    ;			// ignore non-option argument ???
  else {
    print_usage();
    exit(EXIT_FAILURE);
  }

  server = new Server(EPLAYMIDI);
  reg = server->creg->reg;

  signal(SIGHUP, cleanup);
  signal(SIGINT, cleanup);
  signal(SIGTERM, cleanup);
  signal(SIGABRT, cleanup);
  signal(SIGQUIT, cleanup);
  signal(SIGPIPE, cleanup);
  signal(SIGCHLD, cleanup);

  init_inst_name();

  xwin = new XWin(argc, argv);
  xwin->main();

  return EXIT_SUCCESS;
}


#if 0
void DrawProgram(int ch, int prg, int bnk, int map)
{
  switch(map) {
  case 0:
    if(reg->sc88pro) {
      no_str[0] = '8';
      no_str[1] = 'P';
    } else {
      if(reg->sc88) {
	no_str[0] = no_str[1] = '8';
      } else {
	if(reg->sc55) {
	  no_str[0] = no_str[1] = '5';
	} else {
	  no_str[0] = no_str[1] = '5';	// ???
	}
      }
    }
    break;
  case 1: // SC-55 MAP
    no_str[0] = no_str[1] = '5';
    break;
  case 2: // SC-88 MAP
    no_str[0] = no_str[1] = '8';
    break;
  case 3: // SC-88Pro MAP
    no_str[0] = '8';
    no_str[1] = 'P';
    break;
  default: // ???
    no_str[0] = no_str[1] = '?';
    break;
  }
  no_str[2] = '\0';
}
#endif


#if 0
void DrawTrackName(const char* text)
{
  static ch = 0;

  XSetForeground(d, XGC, background);
  XFillRectangle(d, w, XGC,
		 X_TRACK_NAME, Y_CH(ch),
		 X_TRACK_NAME_END-X_TRACK_NAME+1, Y_CH(ch+1)-Y_CH(ch));
  XSetFont(d, XGC, xts->fid);
  XSetForeground(d, XGC, black);
  XDrawStringK(d, w, XGC, 
	       X_TRACK_NAME, Y_CH(ch), text, strlen((char*) text));
  ch++;
}
#endif


/*
 * clean up routine
 */
static void cleanup(int sig)
{
  if(xwin != NULL)
    delete xwin;
  delete server;
  exit((sig > 0) ? EXIT_FAILURE : EXIT_SUCCESS);	// ???
}


/*
 * exit action callback
 */
void ExitAC(Widget /*w*/, XEvent* /*event*/, String* /*params*/,
	    Cardinal* /*num*/)
{
  cleanup(0);
}


/*
 * control callback
 */
void ControlCB(Widget /*w*/, const char* command, XtPointer)
{
  Boolean state;
  if(!strcmp("play", command)) {
    XtVaGetValues(xwin->pauseButton, XtNstate, &state, NULL);
    if(state) {
      Control("continue");
      XtVaSetValues(xwin->pauseButton, XtNstate, False, NULL);
    } else
      Control("play");
  } else if(!strcmp("pause", command)) {
    XtVaGetValues(xwin->pauseButton, XtNstate, &state, NULL);
    if(state)
      Control("pause");
    else
      Control("continue");
  } else if(!strcmp("slow", command)) {
    XtVaGetValues(xwin->backButton, XtNstate, &state, NULL);
    if(state)
      Control("slow");
    else
      Control("normal");
    XtVaGetValues(xwin->fwrdButton, XtNstate, &state, NULL);
    if(state)
      XtVaSetValues(xwin->fwrdButton, XtNstate, False, NULL);
  } else if(!strcmp("fast", command)) {
    XtVaGetValues(xwin->fwrdButton, XtNstate, &state, NULL);
    if(state)
      Control("fast");
    else
      Control("normal");
    XtVaGetValues(xwin->backButton, XtNstate, &state, NULL);
    if(state)
      XtVaSetValues(xwin->backButton, XtNstate, False, NULL);
  } else
    Control(command);
}


/*
 * control action callback
 */
void ControlAC(Widget /*w*/, XEvent* /*event*/, String* params, Cardinal* num)
{
  if(*num != 1)
    fprintf(stderr, "invalid argument to `Control()'.\n");
  else
    Control(params[0]);
}


/*
 * control sequencer
 */
void Control(const char* command)
{
  static unsigned char mvol = 127;	// ???

  if(!strcmp("fast", command))
    server->skew(67.0);
  else if(!strcmp("normal", command))
    server->skew(100.0);
  else if(!strcmp("slow", command))
    server->skew(150.0);
  else if(!strcmp("play", command))
    server->play();
  else if(!strcmp("stop", command))
    server->stop();
  else if(!strcmp("pause", command))
    server->pause();
  else if(!strcmp("continue", command))
    server->unpause();
  else if(!strcmp("previous", command))
    server->prev();
  else if(!strcmp("next", command))
    server->next();
  else if(!strcmp("fade", command))
    server->fadeout();
  else if(!strcmp("clear", command))
    server->clear();
  else if(!strcmp("quit", command)) {
    server->quit();
    cleanup(0);
  } else if(!strcmp("increase", command)) {	// ???
    if(mvol < 127)
      server->parameter('M', ++mvol);
  } else if(!strcmp("decrease", command)) {	// ???
    if(mvol > 0)
      server->parameter('M', --mvol);
  } else if(!strcmp("cursor_up", command)) {
  } else if(!strcmp("cursor_down", command)) {
  } else if(!strcmp("cursor_left", command)) {
  } else if(!strcmp("cursor_right", command)) {
  } else
    fprintf(stderr, "invalid argument to `Control()': `%s\'.\n", command);
}



/*
 * circulate keyboard mode
 */
void KBModeAC(Widget /*w*/, XEvent* /*event*/, String* /*params*/,
	      Cardinal* /*num*/)
{
//int i;
  switch(kbMode) {
  case kbNormal:
    kbMode = kbTrack;
//  for(i = 0; i < MIDI_CHANNEL; i++)
//    track[i]->redraw();
    break;
  case kbTrack:
    kbMode = kbMemo;
//    for(i = 0; i < 12; i++)
//      memo[i]->redraw();
    break;
  case kbMemo:
    kbMode = kbNormal;
//    XSetForeground(d, gc,
//		     allocColor(getResource("keyboard.background")));
//    for(i = 0; i < 12; i++)
//      XFillRectangle(d, w, XGC, X_KEY, Y_CH(i+2), KB_RANGE*4, 16);
//    for(i = 0; i < MIDI_CHANNEL; i++)
//      keyboard[i]->redraw();
    break;
  }
}


/*
 *
 */
void BodyExposeCB(Widget parent, XtPointer body, XtPointer call_data)
{
  if(body != NULL)
    ((Body*)body)->redraw();
}


/*
 *
 */
void LCDDisplayExposeCB(Widget parent, XtPointer lcd, XtPointer call_data)
{
  if(lcd != NULL)
    ((LCDDisplay*)lcd)->redraw();
}


/*
 *
 */
void timer(XtPointer /*p*/, XtIntervalId* /*id*/)
{
  xwin->title->update();
  xwin->body->main();
  xwin->lcd->update();
  xwin->status->update();
  xwin->efxtype->update();
  xwin->timer->update();
  xwin->gmLogo->update();
  xwin->gsLogo->update();
  xwin->xgLogo->update();
}


/*
 *
 */
void FileCB(Widget /*w*/, XtPointer index, XtPointer)
{
  switch((MenuBar::fileMenu)(int)index) {
  case MenuBar::fileMenu_Quit:
    cleanup(0);
    break;
  default:
    break;
  }
}


/*
 *
 */
void InstCB(Widget /*w*/, XtPointer index, XtPointer)
{
#if 0
  switch((MenuBar::instMenu)(int)index) {
  case MenuBar::instMenu_sc55:
    server->instrument(1);
    break;
  case MenuBar::instMenu_sc88:
    server->instrument(2);
    break;
  case MenuBar::instMenu_sc88pro:
    server->instrument(3);
    break;
  case MenuBar::instMenu_xg:
    server->instrument(4);
    break;
  }
#endif
}


