/*
 * shmem.h -- register class
 *
 * Copyright (C) 1995-1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: shmem.h,v 1.8 1998-03-25 08:35:39+09 satoshi Exp $

#ifndef _SHMEM_H_
#define _SHMEM_H_

//#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <cerrno>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/msg.h>

#define SHMEM_SEG_SIZE sizeof(Register)	// the size of shared memory segment

#define STATUS_IDLE   0
#define STATUS_PLAY   1
#define STATUS_PAUSE  2
#define STATUS_FADE   3
#define STATUS_SEARCH 4
#define STATUS_COMP   5

#define MAX_PATH 1024	// maximum length of path name
#define MAX_NAME 255	// maximum length of file name

struct Register {
  // status of sequencer
  unsigned char status;		// status
  unsigned long seqid;		// sequence identifier
  unsigned time;		// elapsed time
  unsigned playtime;		// estimated play time

  // sequencer control
  float skew;			// tempo skew
  unsigned char shuffle;	// shuffle play
  unsigned char repeat;		// repeat mode

  // song data
  int format;			// file format (0:SMF, 1:RCP)
  unsigned timebase;		// time base

  char file[MAX_PATH+MAX_NAME+1];	// file name
  char title[64+1];		// title
  char memo[360/*336*/+1];	// memo (12x28)
  char name[32][20+1];		// track name
  char text[20+1];		// comment
  
  // data specification
  unsigned char gm;		// General MIDI On
  unsigned char gs;		// GS Reset
  unsigned char xg;		// XG System ON

  unsigned char sc55;		// Roland SC-55
  unsigned char sc88;		// Roland SC-88
  unsigned char sc88pro;	// Roland SC-88Pro

  // parameters
  unsigned char vel[32];	// velocity
  unsigned char _vel[32];	// velocity id (for xeplaymidi)
  unsigned char vol[32];	// volume
  unsigned char expr[32];	// expression
  unsigned char mod[32];	// modulation
  unsigned char pan[32];	// panpot
  unsigned char prg[32];	// program
  unsigned char bnk[32];	// bank
  unsigned char map[32];	// map
  unsigned char bend[32];	// bender wheel
  unsigned char rev[32];	// reverb
  unsigned char cho[32];	// chorus
  unsigned char delay[32];	// delay

  unsigned char mvol;		// master volume
  unsigned char mpan;		// master panpot

  unsigned char tempo;		// tempo
  unsigned step;		// step
  unsigned bar;			// bar
  char keysig[10+1];		// key signature

  unsigned char kb[32][128];	// keyboard (0:note off, 1:note on)
  unsigned char rhythm[32];	// use for rhythm part

  char patch[16];		// patch name (SC-88)
  char userdrum[2][12+1];	// user drum set name (SC-88)
  unsigned char efx_type[2];	// EFX type (SC-88Pro)

  // LCD (SC-55)
  unsigned char page;			// Display Page
  unsigned char lcd[32];		// Displayed Letter
  unsigned char lcd_dd[10*16*16];	// Displayed Dot Data
  unsigned char lcd_time;		// Display Time
};


class CRegister {
public:
  volatile Register* reg;

private:
  const key_t key;
  int id;	// shared memory id
  void* addr;	// start address of shared memory segment
  bool create_flag;
  bool success;

public:
  CRegister() : reg(NULL), key(0x51374025), addr(NULL), create_flag(false),
  success(false) {
    errno = 0;
    if((id = shmget(key, sizeof(Register), 
		    (IPC_CREAT | IPC_EXCL | 0600))) < 0) {
      if(errno != EEXIST)
	perror("shmget");
      errno = 0;
      // attach to already created shared memory segment
      if((id = shmget(key, sizeof(Register), 0600)) < 0)
	perror("shmget");
      else
	success = true;
    } else {	// create and initialize shared memory segment by myself
      create_flag = true;
      success = true;
    }
    if(success) {
      addr = shmat(id, 0, 0);
      if(addr == (void *) -1) {
	perror("shmat");
	success = false;
      }
      reg = (Register*)addr;
      if(create_flag) {
	initialize();
	reg->status = STATUS_IDLE;
	reg->seqid = 0;
      }
    }
  }

  ~CRegister() {
    // detach the shared memory segment
    if(shmdt((char*)addr) != 0)
      perror("shmdt");
    struct shmid_ds buf;
    shmctl(id, IPC_STAT, &buf);
    if(buf.shm_nattch == 0) {
      // destroy the id of the shared memory segment
      if(shmctl(id, IPC_RMID, NULL) != 0)
	perror("shmctl");
    }
  }

  bool succeed(void) {
    return success;
  }

  bool create(void) {
    return create_flag;
  }

  void initialize(void) {
    int i, j;
    for(i = 0; i < MAX_PATH+MAX_NAME+1; i++)
      reg->file[i] = '\0';
    for(i = 0; i < 64+1; i++)
      reg->title[i] = '\0';
    for(i = 0; i < 360+1; i++)
      reg->memo[i] = '\0';
    for(i = 0; i < 32; i++) {
      reg->vel[i] = 0;
      reg->_vel[i] = 0;
      reg->vol[i] = 127;
      reg->expr[i] = 127;
      reg->pan[i] = 64;
      reg->bend[i] = 64;
      reg->mod[i] = 0;
      reg->prg[i] = 0;
      reg->bnk[i] = 0;
      reg->map[i] = 0;
      reg->rev[i] = 0;
      reg->cho[i] = 0;
      reg->delay[i] = 0;
      reg->rhythm[i] = (i == 9 || i == 25) ? 1 : 0;
      for(j = 0; j < 128; j++)
	reg->kb[i][j] = 0;
    }
    reg->page = 0;
    reg->lcd_time = 6;	// 2.88 sec
    for(i = 0; i < 10*16*16; i++)
      reg->lcd_dd[i] = 0;
    memcpy((void*)reg->lcd, "- SOUND Canvas -                 ", 32);
    strcpy((char*)reg->keysig, "");
    for(i = 0; i < 20; i++)
      reg->text[i] = ' ';
    reg->text[20] = '\0';
    for(j = 0; j < 32; j++) {
      for(i = 0; i < 20; i++)
	reg->name[j][i] = ' ';
      reg->name[j][20] = '\0';
    }
    reg->time = 0;
    reg->playtime = 0;
    reg->tempo = 0;
    reg->mvol = 127;
    reg->mpan = 64;
    reg->gm = false;
    reg->gs = false;
    reg->xg = false;
    reg->sc88pro = false;
    reg->sc88 = false;
    reg->sc55 = false;
    reg->skew = 100.0;
    reg->shuffle = 0;
    reg->repeat = 0;
    reg->step = 0;
    reg->bar = 0;
    reg->format = 0;
    reg->timebase = 48;		// ???
    memcpy((char*)reg->patch, "- SOUND Canvas -", 16);
    for(i = 0; i < 12; i++)
      reg->userdrum[0][i] = '\0';
    for(i = 0; i < 12; i++)
      reg->userdrum[1][i] = '\0';
    reg->efx_type[0] = 0;
    reg->efx_type[1] = 0;
  }
};


#endif /* !_SHMEM_H_ */
