/*
 * server.cc -- client/server interface
 *
 * Copyright (C) 1997,1998 Satoshi KURAMOCHI <satoshi@ueda.info.waseda.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

// $Id: server.cc,v 1.7 1998-03-25 08:18:11+09 satoshi Exp $

#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <cctype>
#include <cerrno>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "shmem.h"
#include "server.h"

#define SERVICE	"eplaymidi"
#define DEFAULT_PORT 10206	// ???

static const char* lockfile = "/tmp/eplaymidi.pid";


/*
 *
 */
bool server_exist(void)
{
  int fd;
  pid_t pid = -1;
  if((fd = open(lockfile, O_RDONLY, 0)) != -1) {
    char buf[16];
    if(read(fd, buf, sizeof(buf)) > 0)
      pid = atoi(buf);
    close(fd);
    if(pid <= 0 || /*getpgid(pid)*/ kill(pid, 0) == -1) {
      // remove stale shm & msq ???
      unlink(lockfile);
      fprintf(stderr, "stale lock file removed.\n");
      return false;
    } else
      return true;
  }
  return false;
}


/*
 * Server
 */
Server::Server(const char* argv0)
: is_server(false), creg(NULL)
{
  if(argv0 != NULL) {
    if(!server_exist()) {
      // start the server
      int pid;
      switch((pid = fork())) {
      case -1:
	perror("fork");
	exit(EXIT_FAILURE);
	break;
      case 0: // server process
	errno = 0;
	execlp(argv0, argv0, "--daemon", NULL);
	perror("execlp");
	exit(EXIT_FAILURE);
      default: // client process
	{
	  // wait for the server's setup ???
	  int lock_fd;
	  errno = 0;
	  while((lock_fd = open(lockfile, O_RDONLY, 0)) == -1) {
	    if(errno != ENOENT) {
	      perror("open");
	      exit(EXIT_FAILURE);
	    }
	    usleep(100*1000);	// 100ms
	    errno = 0;
	  }
	  close(lock_fd);
	  break;
	}
      }
    }
    // socket initialization
    sockaddr_in remote_addr_in;
    bzero((char*)&remote_addr_in, sizeof(sockaddr_in));
    remote_addr_in.sin_family = AF_INET;
    remote_addr_in.sin_addr.s_addr = INADDR_ANY;
    
    servent* sp;
    if((sp = getservbyname(SERVICE, "tcp")) == NULL) {
#ifdef linux
      remote_addr_in.sin_port = htons(DEFAULT_PORT);
#else
      remote_addr_in.sin_port = DEFAULT_PORT;
#endif
    } else
      remote_addr_in.sin_port = sp->s_port;

    if((sock = socket(AF_INET, SOCK_STREAM, 0)) == -1)
      perror("socket");
    if(connect(sock, (sockaddr*)&remote_addr_in, sizeof(sockaddr_in)) == -1) {
      perror("connect");
      close(sock);
    }
    // shared memory initialization
    key_t key = ftok((char*)lockfile, 1);
    creg = new CRegister();	// ???
  }
}


Server::~Server()
{
  if(!is_server)
    close(sock);
  delete creg;
}


/*
 * remove lock file
 */
void removelock(void)
{
  unlink(lockfile);
}


#define BUFSIZE	4096


/*
 * send a message
 */
bool Server::send(const char* msg)
{
  if(write(sock, msg, strlen(msg)) == -1) {
    perror("write");
    return false;
  } else
    return true;
}


/*
 * receive a message
 */
const char* Server::recv(void)
{
  static char buf[BUFSIZ+1];
  static char* p = NULL;
  char* q = NULL;
  int len = 0;

  if(p != NULL && *p != '\0') {
    q = p;
    if((p = strchr(p, '\n')) != NULL) {
      *p = '\0';
      p++;
      return q;
    } else {
      len = strlen(q);
      memmove(buf, q, len+1);
    }
  }
  long size;
  if(q == NULL) {
    p = buf;
    size = BUFSIZ;
  } else {
    p = buf+len;
    size = BUFSIZ-len;
  }
  for(;;) {
    if((len = read(sock, p, size)) == -1) {
      perror("read");
      return (p = NULL);
    }
    p[len] = '\0';
    q = p;
    if((p = strchr(p, '\n')) != NULL) {
      *p = '\0';
      p++;
      return buf;
    } else {
      if((size -= len) <= 0) {
	fprintf(stderr, "buffer overflow\n");
	return (p = NULL);
      }
      p = q+len;
    }
  }

#if 0
  static char buf[BUFSIZ];
  static char* p = NULL;
  char* q;

  if(p == NULL || *p == '\0') {
    int len;
    if((len = read(sock, &buf, BUFSIZ)) == -1) {
      perror("read");
      p = NULL;
      return p;
    }
    buf[len] = '\0';
    q = buf;
    if((p = strchr(q, '\n')) != NULL) {
      *p = '\0';
      p++;
    } else
      p = NULL;
  } else {
    q = p;
    if((p = strchr(p, '\n')) != NULL) {
      *p = '\0';
      p++;
    } else
      p = NULL;
  }
  return q;
#endif
}


/*
 * add single file to the program
 */
unsigned long Server::add(const char* filename, const char** options)
{
  char* msg = new char[BUFSIZ];
  strcpy(msg, "+");
  int i;
  for(i = 0; options != NULL && options[i] != NULL; i++) {
    strcat(msg, " ");
    strcat(msg, options[i]);
  }
  strcat(msg, " ");
  strcat(msg, filename);
  strcat(msg, "\n");
  send(msg);
  delete[] msg;
  const char* buf = recv();
  return (buf != NULL) ? atoi(buf) : 0;
}


/*
 * play back
 */
bool Server::play(void)
{
  send("P\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * stop playing
 */
bool Server::stop(void)
{
  send("s\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * make a pause
 */
bool Server::pause(void)
{
  send("p\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * leave a pause
 */
bool Server::unpause(void)
{
  send("c\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * play the previous song
 */
bool Server::prev(void)
{
  send("b\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * play the next song
 */
bool Server::next(void)
{
  send("n\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * skew tempo
 */
bool Server::skew(double x)
{
  char msg[256];
  sprintf(msg, "T%g\n", x);
  send(msg);
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * fade out
 */
bool Server::fadeout(void)
{
  send("f\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * clear the list
 */
bool Server::clear(void)
{
  send("l\n");
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * stop the server
 */
bool Server::quit(void)
{
  send("q\n");
  return true;
}


/*
 * set parameter
 */
bool Server::parameter(char n, int x)
{
  char msg[16];
  sprintf(msg, "%c%d\n", n, x);
  send(msg);
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}


/*
 * get a play list
 */
char** Server::getList(void)
{
  send("L\n");
  char** list;
  const char* buf;
  if((buf = recv()) == NULL)
    return NULL;
  int n = atoi(buf);
  list = new char*[n+1];
  int i;
  for(i = 0; i < n; i++) {
    buf = recv();
    list[i] = strdup(buf);
  }
  list[n] = NULL;
  return list;
}


/*
 * set instrument type
 */
bool Server::instrument(int n)
{
  char msg[3] = { 'I', n, 0 };
  send(msg);
  const char* buf;
  int ret;
  if((buf = recv()) == NULL || (ret = atoi(buf)) != 1)
    return false;
  else
    return true;
}
